"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getAccountingStats = void 0;
const client_1 = require("@prisma/client");
const prisma = new client_1.PrismaClient();
/**
 * Get accounting statistics (Admin only)
 * Returns detailed financial data, charts, and audit logs
 */
const getAccountingStats = async (req, res, next) => {
    try {
        // 1. Parse Date Range
        const { startDate, endDate } = req.query;
        let start, end;
        if (startDate && endDate) {
            start = new Date(startDate);
            end = new Date(endDate);
            // Ensure end date covers the full day
            end.setHours(23, 59, 59, 999);
        }
        else {
            // Default to current month
            const now = new Date();
            start = new Date(now.getFullYear(), now.getMonth(), 1);
            end = new Date(now.getFullYear(), now.getMonth() + 1, 0);
            end.setHours(23, 59, 59, 999);
        }
        // 2. Fetch Validated Invoices (PAID Orders)
        const orders = await prisma.order.findMany({
            where: {
                updatedAt: {
                    gte: start,
                    lte: end,
                },
                paymentStatus: 'PAID',
                status: {
                    not: 'CANCELLED'
                }
            },
            include: {
                items: {
                    include: {
                        product: true
                    }
                }
            },
            orderBy: {
                updatedAt: 'asc' // Important for chart data
            }
        });
        // 3. Initialize Accumulators
        let revenueHT = 0;
        let revenueTTC = 0;
        let totalTVA = 0;
        let tva19 = 0;
        let tva7 = 0;
        let tva0 = 0;
        let totalTimbre = 0;
        let totalCost = 0;
        // Chart Data Accumulators
        const dailyRevenue = {};
        const productPerformance = {};
        // 4. Calculate Totals & Process Data
        const invoices = orders.map(order => {
            // Revenue
            const orderHT = order.fiscalTotalHT ? Number(order.fiscalTotalHT) : Number(order.totalNetHT);
            const orderTTC = Number(order.netAPayer);
            revenueHT += orderHT;
            revenueTTC += orderTTC;
            // TVA Breakdown
            tva19 += Number(order.tva19Amount || 0);
            tva7 += Number(order.tva7Amount || 0);
            tva0 += Number(order.tva0Amount || 0);
            totalTVA += Number(order.fiscalTotalTVA || order.totalTVA);
            // Timbre Fiscal
            totalTimbre += Number(order.timbreFiscal || 0);
            // Cost & Margin
            let orderCost = 0;
            order.items.forEach(item => {
                const costPrice = Number(item.product.costPrice || 0);
                orderCost += costPrice * item.quantity;
                totalCost += costPrice * item.quantity;
                // Product Performance
                if (!productPerformance[item.productId]) {
                    productPerformance[item.productId] = {
                        name: item.productName,
                        revenue: 0,
                        cost: 0,
                        quantity: 0
                    };
                }
                productPerformance[item.productId].revenue += Number(item.totalHT);
                productPerformance[item.productId].cost += costPrice * item.quantity;
                productPerformance[item.productId].quantity += item.quantity;
            });
            // Daily Revenue Chart Data
            const dateKey = order.updatedAt.toISOString().split('T')[0];
            if (!dailyRevenue[dateKey]) {
                dailyRevenue[dateKey] = { ht: 0, ttc: 0 };
            }
            dailyRevenue[dateKey].ht += orderHT;
            dailyRevenue[dateKey].ttc += orderTTC;
            // Return simplified invoice object for Audit Mode
            return {
                id: order.id,
                number: order.orderNumber,
                date: order.updatedAt,
                customer: order.customerName,
                totalHT: orderHT,
                totalTVA: Number(order.fiscalTotalTVA || order.totalTVA),
                totalTTC: orderTTC,
                status: order.status
            };
        });
        // 5. Calculate Margins
        const grossMargin = revenueHT - totalCost;
        const marginPercentage = revenueHT > 0 ? (grossMargin / revenueHT) * 100 : 0;
        // 6. Volume Indicators
        const transactionCount = orders.length;
        const averageBasket = transactionCount > 0 ? revenueTTC / transactionCount : 0;
        // 7. Format Chart Data
        const revenueChart = Object.entries(dailyRevenue).map(([date, values]) => ({
            date,
            ht: Math.round(values.ht * 1000) / 1000,
            ttc: Math.round(values.ttc * 1000) / 1000
        })).sort((a, b) => a.date.localeCompare(b.date));
        const topProducts = Object.values(productPerformance)
            .map(p => ({
            name: p.name,
            revenue: Math.round(p.revenue * 1000) / 1000,
            margin: Math.round((p.revenue - p.cost) * 1000) / 1000,
            quantity: p.quantity
        }))
            .sort((a, b) => b.revenue - a.revenue)
            .slice(0, 10);
        res.json({
            success: true,
            data: {
                period: {
                    start: start.toISOString(),
                    end: end.toISOString()
                },
                revenue: {
                    ht: Math.round(revenueHT * 1000) / 1000,
                    ttc: Math.round(revenueTTC * 1000) / 1000,
                    currency: 'TND'
                },
                tva: {
                    total: Math.round(totalTVA * 1000) / 1000,
                    breakdown: {
                        rate19: Math.round(tva19 * 1000) / 1000,
                        rate7: Math.round(tva7 * 1000) / 1000,
                        rate0: Math.round(tva0 * 1000) / 1000
                    },
                    currency: 'TND'
                },
                fiscal: {
                    timbreTotal: Math.round(totalTimbre * 1000) / 1000,
                    currency: 'TND'
                },
                profitability: {
                    totalCost: Math.round(totalCost * 1000) / 1000,
                    grossMargin: Math.round(grossMargin * 1000) / 1000,
                    marginPercentage: Math.round(marginPercentage * 100) / 100,
                    currency: 'TND'
                },
                volume: {
                    invoices: transactionCount,
                    orders: transactionCount,
                    averageBasketTTC: Math.round(averageBasket * 1000) / 1000
                },
                charts: {
                    revenue: revenueChart,
                    topProducts: topProducts
                },
                invoices: invoices // For Audit Mode
            }
        });
    }
    catch (error) {
        next(error);
    }
};
exports.getAccountingStats = getAccountingStats;
//# sourceMappingURL=accounting.controller.js.map