"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reorderBannerSlides = exports.deleteBannerSlide = exports.updateBannerSlide = exports.createBannerSlide = exports.getBannerSlide = exports.getAllBannerSlidesAdmin = exports.getAllBannerSlides = void 0;
const client_1 = require("@prisma/client");
const catalogContext_js_1 = require("../utils/catalogContext.js");
const prisma = new client_1.PrismaClient();
/**
 * Get all banner slides (public - for customer shop)
 */
const getAllBannerSlides = async (req, res) => {
    try {
        const slides = await prisma.bannerSlide.findMany({
            where: {
                isActive: true,
            },
            orderBy: {
                displayOrder: 'asc',
            },
            include: {
                linkedProduct: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
                linkedCategory: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
            },
        });
        res.json({
            success: true,
            data: slides,
        });
    }
    catch (error) {
        console.error('Error fetching banner slides:', error);
        res.status(500).json({
            success: false,
            error: {
                message: 'Failed to fetch banner slides',
            },
        });
    }
};
exports.getAllBannerSlides = getAllBannerSlides;
/**
 * Get all banner slides (admin - includes inactive)
 */
const getAllBannerSlidesAdmin = async (req, res) => {
    try {
        const catalog = await (0, catalogContext_js_1.resolveCatalogContext)(req, prisma);
        const slides = await prisma.bannerSlide.findMany({
            where: {
                catalogId: catalog.id,
            },
            orderBy: {
                displayOrder: 'asc',
            },
            include: {
                linkedProduct: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
                linkedCategory: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
            },
        });
        res.json({
            success: true,
            data: slides,
        });
    }
    catch (error) {
        console.error('Error fetching banner slides:', error);
        res.status(500).json({
            success: false,
            error: {
                message: 'Failed to fetch banner slides',
            },
        });
    }
};
exports.getAllBannerSlidesAdmin = getAllBannerSlidesAdmin;
/**
 * Get single banner slide
 */
const getBannerSlide = async (req, res) => {
    try {
        const { id } = req.params;
        const slide = await prisma.bannerSlide.findUnique({
            where: { id },
            include: {
                linkedProduct: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
                linkedCategory: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
            },
        });
        if (!slide) {
            return res.status(404).json({
                success: false,
                error: {
                    message: 'Banner slide not found',
                },
            });
        }
        res.json({
            success: true,
            data: slide,
        });
    }
    catch (error) {
        console.error('Error fetching banner slide:', error);
        res.status(500).json({
            success: false,
            error: {
                message: 'Failed to fetch banner slide',
            },
        });
    }
};
exports.getBannerSlide = getBannerSlide;
/**
 * Create banner slide
 */
const createBannerSlide = async (req, res) => {
    try {
        const { title, subtitle, buttonText, slideType, imageUrl, backgroundColor, textColor, linkType, linkedProductId, linkedCategoryId, displayOrder, isActive, duration, } = req.body;
        // Validation
        if (!slideType || !['IMAGE', 'TEXT'].includes(slideType)) {
            return res.status(400).json({
                success: false,
                error: {
                    message: 'Invalid slide type. Must be IMAGE or TEXT',
                },
            });
        }
        if (slideType === 'IMAGE' && !imageUrl) {
            return res.status(400).json({
                success: false,
                error: {
                    message: 'Image URL is required for IMAGE slide type',
                },
            });
        }
        if (slideType === 'TEXT' && !backgroundColor) {
            return res.status(400).json({
                success: false,
                error: {
                    message: 'Background color is required for TEXT slide type',
                },
            });
        }
        const slide = await prisma.bannerSlide.create({
            data: {
                title,
                subtitle,
                buttonText,
                slideType,
                imageUrl,
                backgroundColor,
                textColor: textColor || '#FFFFFF',
                linkType: linkType || null,
                linkedProductId: linkedProductId || null,
                linkedCategoryId: linkedCategoryId || null,
                displayOrder: displayOrder || 0,
                isActive: isActive !== undefined ? isActive : true,
                duration: duration || 5,
                catalogId: (await (0, catalogContext_js_1.resolveCatalogContext)(req, prisma)).id,
            },
            include: {
                linkedProduct: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
                linkedCategory: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
            },
        });
        res.status(201).json({
            success: true,
            data: slide,
        });
    }
    catch (error) {
        console.error('Error creating banner slide:', error);
        res.status(500).json({
            success: false,
            error: {
                message: 'Failed to create banner slide',
            },
        });
    }
};
exports.createBannerSlide = createBannerSlide;
/**
 * Update banner slide
 */
const updateBannerSlide = async (req, res) => {
    try {
        const { id } = req.params;
        const { title, subtitle, buttonText, slideType, imageUrl, backgroundColor, textColor, linkType, linkedProductId, linkedCategoryId, displayOrder, isActive, duration, } = req.body;
        const slide = await prisma.bannerSlide.update({
            where: { id },
            data: {
                title,
                subtitle,
                buttonText,
                slideType,
                imageUrl,
                backgroundColor,
                textColor,
                linkType,
                linkedProductId,
                linkedCategoryId,
                displayOrder,
                isActive,
                duration,
            },
            include: {
                linkedProduct: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
                linkedCategory: {
                    select: {
                        id: true,
                        name: true,
                        slug: true,
                    },
                },
            },
        });
        res.json({
            success: true,
            data: slide,
        });
    }
    catch (error) {
        console.error('Error updating banner slide:', error);
        res.status(500).json({
            success: false,
            error: {
                message: 'Failed to update banner slide',
            },
        });
    }
};
exports.updateBannerSlide = updateBannerSlide;
/**
 * Delete banner slide
 */
const deleteBannerSlide = async (req, res) => {
    try {
        const { id } = req.params;
        await prisma.bannerSlide.delete({
            where: { id },
        });
        res.json({
            success: true,
            message: 'Banner slide deleted successfully',
        });
    }
    catch (error) {
        console.error('Error deleting banner slide:', error);
        res.status(500).json({
            success: false,
            error: {
                message: 'Failed to delete banner slide',
            },
        });
    }
};
exports.deleteBannerSlide = deleteBannerSlide;
/**
 * Reorder banner slides
 */
const reorderBannerSlides = async (req, res) => {
    try {
        const { slides } = req.body; // Array of { id, displayOrder }
        if (!Array.isArray(slides)) {
            return res.status(400).json({
                success: false,
                error: {
                    message: 'Slides must be an array',
                },
            });
        }
        // Update all slides in a transaction
        await prisma.$transaction(slides.map((slide) => prisma.bannerSlide.update({
            where: { id: slide.id },
            data: { displayOrder: slide.displayOrder },
        })));
        res.json({
            success: true,
            message: 'Banner slides reordered successfully',
        });
    }
    catch (error) {
        console.error('Error reordering banner slides:', error);
        res.status(500).json({
            success: false,
            error: {
                message: 'Failed to reorder banner slides',
            },
        });
    }
};
exports.reorderBannerSlides = reorderBannerSlides;
//# sourceMappingURL=banner.controller.js.map