"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.updateCatalog = exports.createCatalog = exports.getCatalogById = exports.getCatalogs = void 0;
const database_js_1 = __importDefault(require("../config/database.js"));
const errorHandler_js_1 = require("../middleware/errorHandler.js");
const formatCatalog = (catalog) => {
    const { _count, ...rest } = catalog;
    return {
        ...rest,
        stats: {
            categories: _count?.categories ?? 0,
            products: _count?.products ?? 0,
            banners: _count?.bannerSlides ?? 0,
            customers: _count?.catalogCustomers ?? 0,
        },
    };
};
const prepareCatalogData = (payload) => {
    const data = {};
    if (typeof payload.name === 'string' && payload.name.trim()) {
        data.name = payload.name.trim();
    }
    if (typeof payload.slug === 'string' && payload.slug.trim()) {
        data.slug = payload.slug.trim();
    }
    if (payload.description !== undefined) {
        data.description = typeof payload.description === 'string'
            ? payload.description.trim()
            : null;
    }
    if (payload.logoUrl !== undefined) {
        data.logoUrl = typeof payload.logoUrl === 'string' ? payload.logoUrl.trim() : null;
    }
    if (payload.primaryColor !== undefined) {
        data.primaryColor = typeof payload.primaryColor === 'string'
            ? payload.primaryColor.trim()
            : null;
    }
    if (payload.isActive !== undefined) {
        data.isActive = Boolean(payload.isActive);
    }
    if (payload.isDefault !== undefined) {
        data.isDefault = Boolean(payload.isDefault);
    }
    return data;
};
const getCatalogs = async (req, res, next) => {
    try {
        const catalogs = await database_js_1.default.catalog.findMany({
            orderBy: { name: 'asc' },
            include: {
                _count: {
                    select: {
                        categories: true,
                        products: true,
                        bannerSlides: true,
                        catalogCustomers: true,
                    },
                },
            },
        });
        res.json({
            success: true,
            data: catalogs.map(formatCatalog),
        });
    }
    catch (error) {
        next(error);
    }
};
exports.getCatalogs = getCatalogs;
const getCatalogById = async (req, res, next) => {
    try {
        const { id } = req.params;
        const catalog = await database_js_1.default.catalog.findUnique({
            where: { id },
            include: {
                _count: {
                    select: {
                        categories: true,
                        products: true,
                        bannerSlides: true,
                        catalogCustomers: true,
                    },
                },
            },
        });
        if (!catalog) {
            throw (0, errorHandler_js_1.createError)('Catalog not found', 404);
        }
        res.json({
            success: true,
            data: formatCatalog(catalog),
        });
    }
    catch (error) {
        next(error);
    }
};
exports.getCatalogById = getCatalogById;
const createCatalog = async (req, res, next) => {
    try {
        const catalogData = prepareCatalogData(req.body);
        if (!catalogData.name) {
            throw (0, errorHandler_js_1.createError)('Name is required', 400);
        }
        if (!catalogData.slug) {
            throw (0, errorHandler_js_1.createError)('Slug is required', 400);
        }
        const existingSlug = await database_js_1.default.catalog.findUnique({ where: { slug: catalogData.slug } });
        if (existingSlug) {
            throw (0, errorHandler_js_1.createError)('Slug already in use', 409);
        }
        const catalog = await database_js_1.default.$transaction(async (tx) => {
            if (catalogData.isDefault) {
                await tx.catalog.updateMany({
                    where: { isDefault: true },
                    data: { isDefault: false },
                });
            }
            return tx.catalog.create({ data: catalogData });
        });
        res.status(201).json({
            success: true,
            message: 'Catalog created successfully',
            data: catalog,
        });
    }
    catch (error) {
        next(error);
    }
};
exports.createCatalog = createCatalog;
const updateCatalog = async (req, res, next) => {
    try {
        const { id } = req.params;
        const existingCatalog = await database_js_1.default.catalog.findUnique({ where: { id } });
        if (!existingCatalog) {
            throw (0, errorHandler_js_1.createError)('Catalog not found', 404);
        }
        const catalogData = prepareCatalogData(req.body);
        if (catalogData.slug && catalogData.slug !== existingCatalog.slug) {
            const slugTaken = await database_js_1.default.catalog.findUnique({ where: { slug: catalogData.slug } });
            if (slugTaken) {
                throw (0, errorHandler_js_1.createError)('Slug already in use', 409);
            }
        }
        if (Object.keys(catalogData).length === 0) {
            throw (0, errorHandler_js_1.createError)('No update fields provided', 400);
        }
        const catalog = await database_js_1.default.$transaction(async (tx) => {
            if (catalogData.isDefault) {
                await tx.catalog.updateMany({
                    where: {
                        isDefault: true,
                        NOT: { id },
                    },
                    data: { isDefault: false },
                });
            }
            return tx.catalog.update({
                where: { id },
                data: catalogData,
            });
        });
        res.json({
            success: true,
            message: 'Catalog updated successfully',
            data: catalog,
        });
    }
    catch (error) {
        next(error);
    }
};
exports.updateCatalog = updateCatalog;
//# sourceMappingURL=catalog.controller.js.map