"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFilePathFromUrl = exports.getFilenameFromUrl = exports.getFileUrl = exports.deleteFile = exports.upload = void 0;
const multer_1 = __importDefault(require("multer"));
const path_1 = __importDefault(require("path"));
const fs_1 = __importDefault(require("fs"));
// In CommonJS, __dirname is globally available
// const __filename = fileURLToPath(import.meta.url);
// const __dirname = dirname(__filename);
// Ensure uploads directory exists
const uploadsDir = path_1.default.join(__dirname, '../../uploads');
const categoriesDir = path_1.default.join(uploadsDir, 'categories');
const productsDir = path_1.default.join(uploadsDir, 'products');
[uploadsDir, categoriesDir, productsDir].forEach(dir => {
    if (!fs_1.default.existsSync(dir)) {
        fs_1.default.mkdirSync(dir, { recursive: true });
    }
});
// Configure storage
const storage = multer_1.default.diskStorage({
    destination: (req, file, cb) => {
        // Determine destination based on upload type
        const uploadType = req.body.uploadType || 'products';
        const dest = uploadType === 'categories' ? categoriesDir : productsDir;
        cb(null, dest);
    },
    filename: (req, file, cb) => {
        // Generate unique filename
        const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
        const ext = path_1.default.extname(file.originalname);
        const name = path_1.default.basename(file.originalname, ext)
            .replace(/[^a-z0-9]/gi, '-')
            .toLowerCase();
        cb(null, `${name}-${uniqueSuffix}${ext}`);
    },
});
// File filter
const fileFilter = (req, file, cb) => {
    // Accept images only
    const allowedTypes = /jpeg|jpg|png|gif|webp/;
    const extname = allowedTypes.test(path_1.default.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);
    if (mimetype && extname) {
        cb(null, true);
    }
    else {
        cb(new Error('Only image files are allowed (jpeg, jpg, png, gif, webp)'));
    }
};
// Multer upload configuration
exports.upload = (0, multer_1.default)({
    storage,
    fileFilter,
    limits: {
        fileSize: 5 * 1024 * 1024, // 5MB max file size
    },
});
// Helper function to delete file
const deleteFile = (filePath) => {
    return new Promise((resolve, reject) => {
        fs_1.default.unlink(filePath, (err) => {
            if (err && err.code !== 'ENOENT') {
                // Ignore if file doesn't exist
                reject(err);
            }
            else {
                resolve();
            }
        });
    });
};
exports.deleteFile = deleteFile;
// Helper function to get file URL
const getFileUrl = (filename, type = 'products') => {
    const baseUrl = process.env.API_URL || 'http://localhost:3001';
    return `${baseUrl}/uploads/${type}/${filename}`;
};
exports.getFileUrl = getFileUrl;
// Helper function to extract filename from URL
const getFilenameFromUrl = (url) => {
    const match = url.match(/\/uploads\/(categories|products)\/(.+)$/);
    return match ? match[2] : null;
};
exports.getFilenameFromUrl = getFilenameFromUrl;
// Helper function to get file path from URL
const getFilePathFromUrl = (url) => {
    const match = url.match(/\/uploads\/(categories|products)\/(.+)$/);
    if (!match)
        return null;
    const [, type, filename] = match;
    const dir = type === 'categories' ? categoriesDir : productsDir;
    return path_1.default.join(dir, filename);
};
exports.getFilePathFromUrl = getFilePathFromUrl;
//# sourceMappingURL=upload.service.js.map