"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.resolveCatalogContext = exports.ensureCustomerAssignedToCatalog = void 0;
const errorHandler_js_1 = require("../middleware/errorHandler.js");
const getStringValue = (value) => {
    if (typeof value === 'string' && value.trim().length > 0) {
        return value.trim();
    }
    return undefined;
};
const ensureCustomerAssignedToCatalog = async (prisma, customerId, catalogId) => {
    const assignment = await prisma.catalogCustomer.findFirst({
        where: {
            catalogId,
            customerId,
        },
    });
    if (!assignment) {
        throw (0, errorHandler_js_1.createError)('Customer does not belong to this catalog', 403);
    }
    return assignment;
};
exports.ensureCustomerAssignedToCatalog = ensureCustomerAssignedToCatalog;
const resolveCatalogContext = async (req, prisma, options = {}) => {
    const { required = true, fallbackToDefault = true, allowBody = false, useCustomerAssignment = true, requireCustomerAssignment = false, } = options;
    const headerCatalogId = getStringValue(req.header('x-catalog-id'));
    const headerCatalogSlug = getStringValue(req.header('x-catalog-slug'));
    const queryCatalogId = getStringValue(req.query.catalogId);
    const queryCatalogSlug = getStringValue(req.query.catalogSlug);
    const bodyCatalogId = allowBody ? getStringValue((req.body || {}).catalogId) : undefined;
    const bodyCatalogSlug = allowBody ? getStringValue((req.body || {}).catalogSlug) : undefined;
    const requestedCatalogId = headerCatalogId || bodyCatalogId || queryCatalogId;
    const requestedCatalogSlug = headerCatalogSlug || bodyCatalogSlug || queryCatalogSlug;
    let catalog = null;
    if (requestedCatalogId) {
        catalog = await prisma.catalog.findUnique({ where: { id: requestedCatalogId } });
    }
    if (!catalog && requestedCatalogSlug) {
        catalog = await prisma.catalog.findUnique({ where: { slug: requestedCatalogSlug } });
    }
    const user = req.user;
    if (!catalog && useCustomerAssignment && user?.role === 'CUSTOMER') {
        const assignment = await prisma.catalogCustomer.findFirst({
            where: { customerId: user.id },
            include: { catalog: true },
            orderBy: { assignedAt: 'desc' },
        });
        if (assignment?.catalog) {
            catalog = assignment.catalog;
        }
    }
    if (!catalog && fallbackToDefault) {
        catalog = await prisma.catalog.findFirst({ where: { isDefault: true } });
    }
    if (!catalog) {
        if (required) {
            throw (0, errorHandler_js_1.createError)('Unable to resolve catalog context', 400);
        }
        throw (0, errorHandler_js_1.createError)('Catalog not found', 404);
    }
    if (!catalog.isActive) {
        throw (0, errorHandler_js_1.createError)('Catalog is inactive', 403);
    }
    if (requireCustomerAssignment && user?.role === 'CUSTOMER') {
        await (0, exports.ensureCustomerAssignedToCatalog)(prisma, user.id, catalog.id);
    }
    return {
        id: catalog.id,
        slug: catalog.slug,
        name: catalog.name,
        isDefault: catalog.isDefault,
    };
};
exports.resolveCatalogContext = resolveCatalogContext;
//# sourceMappingURL=catalogContext.js.map