import { Express } from 'express';
import { hooks, HookSystem } from './HookSystem.js';
import fs from 'fs/promises';
import path from 'path';

export interface PluginDefinition {
    name: string;
    version: string;
    description?: string;
    init: (app: Express, hooks: HookSystem) => Promise<void>;
    navigation?: {
        name: string;
        href: string;
        icon: string;
        roles: string[];
    }[];
}

interface PluginState {
    name: string;
    enabled: boolean;
}

export class PluginManager {
    private plugins: PluginDefinition[] = [];
    private app: Express;
    private stateFile = path.join(process.cwd(), 'data', 'plugins.json');
    private states: Record<string, boolean> = {};

    constructor(app: Express) {
        this.app = app;
    }

    /**
     * Register a plugin manually
     */
    register(plugin: PluginDefinition) {
        this.plugins.push(plugin);
        console.log(`[PluginManager] Registered plugin: ${plugin.name} v${plugin.version}`);
    }

    private async loadStates() {
        try {
            await fs.mkdir(path.dirname(this.stateFile), { recursive: true });
            const data = await fs.readFile(this.stateFile, 'utf-8');
            const parsed = JSON.parse(data) as PluginState[];
            this.states = parsed.reduce((acc, curr) => ({ ...acc, [curr.name]: curr.enabled }), {});
        } catch (error) {
            console.log('[PluginManager] No plugin state file found, creating default.');
            this.states = {};
        }
    }

    private async saveStates() {
        const data = Object.entries(this.states).map(([name, enabled]) => ({ name, enabled }));
        await fs.writeFile(this.stateFile, JSON.stringify(data, null, 2));
    }

    /**
     * Initialize all registered plugins
     */
    private configFile = path.join(process.cwd(), 'data', 'plugin-configs.json');
    private configs: Record<string, any> = {};

    private async loadConfigs() {
        try {
            const data = await fs.readFile(this.configFile, 'utf-8');
            this.configs = JSON.parse(data);
        } catch (error) {
            console.log('[PluginManager] No plugin config file found, creating default.');
            this.configs = {};
        }
    }

    private async saveConfigs() {
        await fs.writeFile(this.configFile, JSON.stringify(this.configs, null, 2));
    }

    async init() {
        await this.loadStates();
        await this.loadConfigs();

        console.log('[PluginManager] Initializing plugins...');
        for (const plugin of this.plugins) {
            // Default to enabled if not in state
            if (this.states[plugin.name] === undefined) {
                this.states[plugin.name] = true;
            }

            if (!this.states[plugin.name]) {
                console.log(`[PluginManager] Skipping disabled plugin: ${plugin.name}`);
                continue;
            }

            try {
                // Initialize default config if missing
                if (!this.configs[plugin.name]) {
                    this.configs[plugin.name] = {};
                }

                // [HARDENING] Route Auditing
                // Wrap app to log route registrations
                const appWrapper = new Proxy(this.app, {
                    get: (target: any, prop: string) => {
                        if (['get', 'post', 'put', 'delete', 'patch', 'use'].includes(prop)) {
                            return (...args: any[]) => {
                                const path = typeof args[0] === 'string' ? args[0] : '(middleware)';
                                console.log(`[PluginManager] 🛡️ Plugin '${plugin.name}' registering: [${prop.toUpperCase()}] ${path}`);
                                return target[prop](...args);
                            };
                        }
                        return target[prop];
                    }
                });

                await plugin.init(appWrapper, hooks);
                console.log(`[PluginManager] initialized: ${plugin.name}`);
            } catch (error) {
                console.error(`[PluginManager] Failed to init ${plugin.name}:`, error);
            }
        }
        await this.saveStates();
        await this.saveConfigs();
    }

    /**
     * Toggle a plugin state
     */
    async togglePlugin(name: string, enabled: boolean) {
        this.states[name] = enabled;
        await this.saveStates();
        console.log(`[PluginManager] Plugin ${name} is now ${enabled ? 'ENABLED' : 'DISABLED'}`);
    }

    /**
     * Get list of registered plugins with status
     */
    getPlugins() {
        return this.plugins.map(p => ({
            ...p,
            status: this.states[p.name] ? 'ACTIVE' : 'INACTIVE'
        }));
    }

    async getPluginConfig(name: string) {
        return this.configs[name] || {};
    }

    async updatePluginConfig(name: string, config: any) {
        this.configs[name] = { ...this.configs[name], ...config };
        await this.saveConfigs();
        console.log(`[PluginManager] Updated config for ${name}`);
    }

    /**
     * Install a plugin from a zip file
     */
    async installPlugin(zipPath: string): Promise<void> {
        return new Promise(async (resolve, reject) => {
            try {
                // Dynamic import to avoid build errors if package is missing during dev
                const AdmZip = (await import('adm-zip')).default;
                const zip = new AdmZip(zipPath);

                // Extract directly to plugins directory
                // We assume the zip contains a folder with the plugin name
                const pluginsDir = path.join(process.cwd(), 'src', 'plugins');
                await fs.mkdir(pluginsDir, { recursive: true });

                zip.extractAllTo(pluginsDir, true);

                // Note: In production we might need to compile the TS or require a restart
                // For now, extraction is enough, user must restart.
                console.log(`[PluginManager] Extracted plugin to ${pluginsDir}`);
                resolve();
            } catch (err) {
                console.error('[PluginManager] Failed to install plugin:', err);
                reject(err);
            }
        });
    }

    /**
     * Get merged navigation from active plugins
     */
    getNavigation() {
        return this.plugins
            .filter(p => this.states[p.name])
            .flatMap(p => p.navigation || []);
    }
}
