import express from 'express';
import cors from 'cors';
import helmet from 'helmet';
import compression from 'compression';
import morgan from 'morgan';
import dotenv from 'dotenv';
import rateLimit from 'express-rate-limit';

// Load environment variables
dotenv.config();

// Import routes
import authRoutes from './routes/auth.routes.js';
import categoryRoutes from './routes/category.routes.js';
import catalogRoutes from './routes/catalog.routes.js';
import productRoutes from './routes/product.routes.js';
import uploadRoutes from './routes/upload.routes.js';
import customerRoutes from './routes/customer.routes.js';
import orderRoutes from './routes/order.routes.js';
import bannerRoutes from './routes/banner.routes.js';
import quotationRoutes from './routes/quotation.routes.js';
import dashboardRoutes from './routes/dashboard.routes.js';
import invoiceRoutes from './routes/invoice.routes.js';
import accountingRoutes from './routes/accounting.routes.js';
import settingsRoutes from './routes/settings.routes.js';
import adminRoutes from './routes/admin.routes.js';
import pluginRoutes from './routes/plugin.routes.js';

// Import middleware
import { errorHandler } from './middleware/errorHandler.js';
import { notFound } from './middleware/notFound.js';

const app = express();
const PORT = process.env.PORT || 3001;

// Plugin System Initialization
import { PluginManager } from './core/PluginManager.js';
import * as taxPlugin from './plugins/tax-tunisia/index.js';

const pluginManager = new PluginManager(app);
// Register Core Plugins
pluginManager.register(taxPlugin);

// Initialize plugins
// Note: We use an IIFE or similar if top-level await isn't supported, 
// but we'll assume the server startup can handle async, or we call init inside listen?
// Better to init before creating routes so plugins can register routes.
// We will call init() immediately but since it is async, we need to handle valid startup.
// For now, let's just trigger it.
// Plugin init is now handled in startServer()

// ============================================
// MIDDLEWARE
// ============================================

// Security
app.use(helmet());

// CORS
const corsOptions = {
  origin:
    [
      'http://localhost:5173',
      'http://localhost:5174',
      'http://localhost:5175',
      ...(process.env.CORS_ORIGIN?.split(',') || [])
    ],
  credentials: true,
};
app.use(cors(corsOptions));

// Body parsing
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Compression
app.use(compression());

// Logging
if (process.env.NODE_ENV === 'development') {
  app.use(morgan('dev'));
} else {
  app.use(morgan('combined'));
}

// Rate limiting
const limiter = rateLimit({
  windowMs: parseInt(process.env.RATE_LIMIT_WINDOW_MS || '900000'), // 15 minutes
  max: parseInt(process.env.RATE_LIMIT_MAX_REQUESTS || '10000'), // Increased from 1000 to 10000 for development
  message: 'Too many requests from this IP, please try again later.',
  standardHeaders: true, // Return rate limit info in the `RateLimit-*` headers
  legacyHeaders: false, // Disable the `X-RateLimit-*` headers
  // Skip rate limiting in development if needed
  skip: (req) => {
    // Skip rate limiting for development environment
    return process.env.NODE_ENV === 'development';
  },
});
app.use('/api/', limiter);

// Static files (uploads)
app.use('/uploads', express.static('uploads'));

// ============================================
// ROUTES
// ============================================

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    environment: process.env.NODE_ENV,
  });
});

// API routes
// Attach to app for controllers
app.set('pluginManager', pluginManager);

// API routes
app.use('/api/auth', authRoutes);
app.use('/api/categories', categoryRoutes);
app.use('/api/catalogs', catalogRoutes);
app.use('/api/products', productRoutes);
app.use('/api/upload', uploadRoutes);
app.use('/api/customers', customerRoutes);
app.use('/api/orders', orderRoutes);
app.use('/api/quotations', quotationRoutes);
app.use('/api/invoices', invoiceRoutes);
app.use('/api/banners', bannerRoutes);
app.use('/api/dashboard', dashboardRoutes);
app.use('/api/admin/accounting', accountingRoutes);
app.use('/api/admin/users', adminRoutes);
app.use('/api/admin/plugins', pluginRoutes); // Plugins Route
app.use('/api/settings', settingsRoutes);

// ============================================
// ERROR HANDLING
// ============================================

app.use(notFound);
app.use(errorHandler);

// ============================================
// START SERVER
// ============================================
// [HARDENING] Deterministic Startup
const startServer = async () => {
  try {
    // Initialize plugins BEFORE listening
    // This ensures routes are verified and hooks are ready
    console.log('[Server] Initializing Plugin System...');
    await pluginManager.init();

    app.listen(PORT, () => {
      console.log(`
╔═══════════════════════════════════════════╗
║   SQB Hardware Store API Server          ║
║   Environment: ${process.env.NODE_ENV?.padEnd(28) || 'development'.padEnd(28)}║
║   Port: ${PORT.toString().padEnd(34)}║
║   Status: Running ✓                       ║
╚═══════════════════════════════════════════╝
      `);
    });
  } catch (err) {
    console.error('🚨 Server Startup Failed:', err);
    process.exit(1);
  }
};

startServer();

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  process.exit(0);
});

process.on('SIGINT', () => {
  console.log('SIGINT signal received: closing HTTP server');
  process.exit(0);
});

export default app;
