
# Plugin Creation Guide

This system supports two types of plugins:
1.  **Backend Plugins**: Server-side logic, hooks, API extensions, and integrations. (Can be uploaded dynamically).
2.  **Frontend Plugins**: Client-side UI components and interface extensions. (Requires rebuild/redeploy).

---

## 1. Backend Plugins
Backend plugins allow you to intercept system logic (Tax, Inventory, Orders) using the `HookSystem`.

### Directory Structure
Location: `backend/src/plugins/<your-plugin-name>/`

```
your-plugin-name/
├── index.ts        # Entry point (Required)
├── logic.ts        # Business logic (Recommended)
└── config.json     # Default configuration (Optional)
```

### `index.ts` Template
Every backend plugin must export specific fields and an `init` function.

```typescript
import { Express } from 'express';
// Adjust path to point to backend/src/core/HookSystem.js
import { HookSystem } from '../../core/HookSystem.js'; 

export const name = 'my-cool-plugin';
export const version = '1.0.0';
export const description = 'Does amazing things with orders';

// Optional: Add a link to the Admin Sidebar
export const navigation = [
    {
        name: 'My Plugin',
        href: '/settings/my-plugin',
        icon: 'Settings', // Lucide React Icon name
        roles: ['ADMIN']
    }
];

export const init = async (app: Express, hooks: HookSystem) => {
    
    // 1. Get Configuration
    const getConfig = async () => {
        // 'pluginManager' is available on app
        const pm = (app.get('pluginManager') as any);
        return await pm.getPluginConfig('my-cool-plugin');
    };

    // 2. Register Hooks
    // Example: Intercept line item calculation
    hooks.on('calculate.line_item', async (context: any, next) => {
        try {
            console.log('Intercepting line item:', context);
            // Modify context.result or context.input here
        } catch (e) {
            console.error(e);
        }
        // ALWAYS call next()
        await next();
    });

    console.log('[MyPlugin] Initialized!');
};
```

### Available Hooks
*   `calculate.line_item`: Modify price/tax for a single item.
*   `calculate.document`: Modify total invoice/order sums.
*   `tva.get_effective_rate`: Change tax rate based on product/catalog.
*   `tva.is_exempt`: Override tax exemption status.

### Packaging for Upload
To upload this plugin via the Admin Dashboard:
1.  Create a folder named `my-cool-plugin`.
2.  Put your `index.ts` (and other files) inside.
3.  Zip the **folder** (not just the files inside).
4.  Upload the `.zip` file in **Settings > Plugins**.

---

## 2. Frontend Plugins
Frontend plugins inject UI components into specific "Slots" in the application.

### Directory Structure
Location: `src/plugins/<your-plugin-name>/`

```
your-plugin-name/
├── index.ts            # Entry point
└── MyComponent.tsx     # Your React Component
```

### `index.ts` Template

```typescript
// Adjust path to src/core/ComponentRegistry
import { registry } from '../../core/ComponentRegistry';
import { MyComponent } from './MyComponent';

export const initMyPlugin = () => {
    // Register your component to a slot
    registry.register('product_actions', MyComponent);
    registry.register('cart_footer', MyComponent);
};
```

### Available Slots
*   `product_actions`: Buttons near "Add to Cart" on product detail.
*   `cart_footer`: Content at the bottom of the floating cart.
*   *(More slots can be added by wrapping code in `ExtensionSlot` components)*

### Activation
Frontend plugins currently require manual activation in `src/main.tsx`:

```typescript
// src/main.tsx
import { initMyPlugin } from './plugins/my-cool-plugin';

// ...
initMyPlugin(); // Call before rendering App
// ...
```

*Note: After adding a frontend plugin, you must rebuild the frontend (`npm run build`) and redeploy.*
