import { Award, TrendingUp, Truck, Headphones } from 'lucide-react';
import BannerSlider from '../components/BannerSlider';
import FeaturedProducts from '../components/FeaturedProducts';
import { Category } from '../types/api';
import { useLanguage } from '../contexts/LanguageContext';

interface HomeScreenProps {
  categories: Category[];
  searchQuery?: string;
  onSearchChange?: (query: string) => void;
  onCategoryClick: (categoryId: string) => void;
  onProductClick?: (productId: string) => void;
}

export default function HomeScreen({ categories, onCategoryClick, onProductClick }: HomeScreenProps) {
  const { t } = useLanguage();

  const handleBannerClick = (slide: any) => {
    // Handle banner click - navigate to linked product or category
    if (slide.linkType === 'CATEGORY' && slide.linkedCategoryId) {
      onCategoryClick(slide.linkedCategoryId);
    } else if (slide.linkType === 'PRODUCT' && slide.linkedProductId && onProductClick) {
      onProductClick(slide.linkedProductId);
    }
  };

  const handleProductClick = (productId: string) => {
    if (onProductClick) {
      onProductClick(productId);
    }
  };

  // Show all root categories
  const displayCategories = categories;

  return (
    <div className="pb-6">
      {/* Categories Section - FIRST as requested */}
      <div className="px-4 sm:px-6 lg:px-8 py-8 bg-white mb-8 border-b border-gray-100">
        <div className="max-w-7xl mx-auto">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-2xl font-bold text-gray-900">
              {t.home?.browseCategories || "Browse Categories"}
            </h2>
          </div>

          {categories.length === 0 ? (
            <div className="text-center py-12">
              <p className="text-gray-500">
                {t.common?.noCategories || "No categories available"}
              </p>
            </div>
          ) : (
            <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-6 gap-4 lg:gap-6">
              {displayCategories.map((category) => (
                <div
                  key={category.id}
                  onClick={() => onCategoryClick(category.id)}
                  className="bg-gray-50 rounded-xl overflow-hidden hover:shadow-md transition-all cursor-pointer group border border-gray-100"
                >
                  <div className="aspect-square bg-white overflow-hidden p-4 flex items-center justify-center">
                    {category.image ? (
                      <img
                        src={category.image}
                        alt={category.name}
                        className="w-full h-full object-contain group-hover:scale-110 transition-transform duration-300"
                      />
                    ) : (
                      <div className="w-12 h-12 text-blue-200">
                        <svg className="w-full h-full" fill="currentColor" viewBox="0 0 24 24"><path d="M10 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V8c0-1.1-.9-2-2-2h-8l-2-2z" /></svg>
                      </div>
                    )}
                  </div>
                  <div className="p-3 text-center bg-white border-t border-gray-100">
                    <h3 className="font-medium text-gray-900 text-sm group-hover:text-blue-600 transition-colors line-clamp-1">
                      {category.name}
                    </h3>
                    <p className="text-xs text-gray-500 mt-0.5">
                      {category.productCount || 0} {t.common?.items || "items"}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>

      {/* Hero Banner - SECOND */}
      <div className="px-4 sm:px-6 lg:px-8 mb-12 max-w-7xl mx-auto">
        <BannerSlider onSlideClick={handleBannerClick} />
      </div>

      {/* Featured Products Section - THIRD */}
      <div className="max-w-7xl mx-auto">
        <FeaturedProducts onProductClick={handleProductClick} />
      </div>

      {/* Why Choose Us Section */}
      <div className="bg-gray-50 py-12 px-4 sm:px-6 lg:px-8 mt-12 border-t border-gray-200">
        <div className="max-w-6xl mx-auto">
          <h2 className="text-2xl font-bold text-gray-900 text-center mb-8">
            {t.home?.whyChooseUs || "Why Choose Us?"}
          </h2>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 lg:gap-8">
            {/* Quality Products */}
            <div className="text-center p-6 bg-white rounded-xl shadow-sm hover:shadow-md transition-shadow">
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Award className="w-6 h-6 text-blue-600" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">
                {t.home?.qualityProducts || "Quality Products"}
              </h3>
              <p className="text-sm text-gray-600">
                {t.home?.qualityProductsDesc || "We offer only the best quality products from trusted brands"}
              </p>
            </div>

            {/* High Ratings */}
            <div className="text-center p-6 bg-white rounded-xl shadow-sm hover:shadow-md transition-shadow">
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <TrendingUp className="w-6 h-6 text-blue-600" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">
                {t.home?.highRatings || "High Ratings"}
              </h3>
              <p className="text-sm text-gray-600">
                {t.home?.highRatingsDesc || "Rated 4.8/5 by thousands of satisfied customers"}
              </p>
            </div>

            {/* Free Delivery */}
            <div className="text-center p-6 bg-white rounded-xl shadow-sm hover:shadow-md transition-shadow">
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Truck className="w-6 h-6 text-blue-600" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">
                {t.home?.freeDelivery || "Free Delivery"}
              </h3>
              <p className="text-sm text-gray-600">
                {t.home?.freeDeliveryDesc || "Free shipping on orders over 200 TND"}
              </p>
            </div>

            {/* Good Support */}
            <div className="text-center p-6 bg-white rounded-xl shadow-sm hover:shadow-md transition-shadow">
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                <Headphones className="w-6 h-6 text-blue-600" />
              </div>
              <h3 className="font-semibold text-gray-900 mb-2">
                {t.home?.goodSupport || "Good Support"}
              </h3>
              <p className="text-sm text-gray-600">
                {t.home?.goodSupportDesc || "24/7 customer support ready to help you"}
              </p>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
}
