import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function diagnose() {
    console.log('=== CATALOG ISOLATION DIAGNOSTIC ===\n');

    // 1. Check catalogs
    const catalogs = await prisma.catalog.findMany();
    console.log(`✓ Found ${catalogs.length} catalogs:`);
    catalogs.forEach(c => console.log(`  - ${c.name} (${c.id}) ${c.isDefault ? '[DEFAULT]' : ''}`));

    // 2. Check customers
    const customers = await prisma.user.findMany({
        where: { role: 'CUSTOMER' },
        include: { catalogAssignments: { include: { catalog: true } } }
    });

    console.log(`\n✓ Found ${customers.length} customers:`);
    customers.forEach(u => {
        const catalog = u.catalogAssignments[0]?.catalog;
        const status = catalog ? `→ ${catalog.name}` : '⚠️  NO CATALOG!';
        console.log(`  - ${u.email} ${status}`);
    });

    // 3. Check categories
    const categories = await prisma.category.findMany({
        include: { catalog: true }
    });

    const nullCatalogCategories = categories.filter(c => !c.catalogId);
    console.log(`\n✓ Categories: ${categories.length} total`);

    // Group by catalog
    const categoriesByCatalog = categories.reduce((acc, cat) => {
        const catalogName = cat.catalog?.name || 'NULL';
        if (!acc[catalogName]) acc[catalogName] = [];
        acc[catalogName].push(cat.name);
        return acc;
    }, {} as Record<string, string[]>);

    Object.entries(categoriesByCatalog).forEach(([catalogName, cats]) => {
        console.log(`  - ${catalogName}: ${cats.length} categories`);
        cats.forEach(name => console.log(`    • ${name}`));
    });

    if (nullCatalogCategories.length > 0) {
        console.log(`  ⚠️  ${nullCatalogCategories.length} categories have NULL catalog_id!`);
    }

    // 4. Check products
    const products = await prisma.product.findMany({
        include: { catalog: true, category: true }
    });

    const nullCatalogProducts = products.filter(p => !p.catalogId);
    console.log(`\n✓ Products: ${products.length} total`);

    // Group by catalog
    const productsByCatalog = products.reduce((acc, prod) => {
        const catalogName = prod.catalog?.name || 'NULL';
        if (!acc[catalogName]) acc[catalogName] = [];
        acc[catalogName].push(`${prod.name} (category: ${prod.category?.name || 'none'})`);
        return acc;
    }, {} as Record<string, string[]>);

    Object.entries(productsByCatalog).forEach(([catalogName, prods]) => {
        console.log(`  - ${catalogName}: ${prods.length} products`);
        prods.forEach(name => console.log(`    • ${name}`));
    });

    if (nullCatalogProducts.length > 0) {
        console.log(`  ⚠️  ${nullCatalogProducts.length} products have NULL catalog_id!`);
    }

    // 5. Summary
    console.log('\n=== SUMMARY ===');
    const issues = [];

    const customersWithoutCatalog = customers.filter(u => !u.catalogAssignments.length);
    if (customersWithoutCatalog.length > 0) {
        issues.push(`${customersWithoutCatalog.length} customers have no catalog assignment`);
        customersWithoutCatalog.forEach(u => console.log(`  ⚠️  ${u.email} has no catalog`));
    }

    if (nullCatalogCategories.length > 0) {
        issues.push(`${nullCatalogCategories.length} categories missing catalog_id`);
    }

    if (nullCatalogProducts.length > 0) {
        issues.push(`${nullCatalogProducts.length} products missing catalog_id`);
    }

    if (issues.length === 0) {
        console.log('✅ No issues found! Catalog isolation should work.');
    } else {
        console.log('❌ Issues found:');
        issues.forEach(issue => console.log(`  - ${issue}`));
        console.log('\nRun fix-catalog-assignments.ts to resolve these issues.');
    }
}

diagnose()
    .catch(console.error)
    .finally(() => prisma.$disconnect());
