#!/bin/bash

# Multi-Catalog Migration Script for TEST Environment
# This script runs the Prisma migration and seeds the default catalog in test containers

set -e  # Exit on error

echo "========================================="
echo "Multi-Catalog TEST Migration Script"
echo "========================================="
echo ""

# Use test docker-compose file
COMPOSE_FILE="docker-compose.test.yml"

# Check if Docker is running
if ! docker ps > /dev/null 2>&1; then
    echo "❌ Error: Docker is not running or you don't have permission to access it."
    echo "Please start Docker and try again."
    exit 1
fi

echo "✅ Docker is running"
echo ""

# Check if test containers are running
if ! docker ps | grep -q smd-backend-test; then
    echo "⚠️  Test containers not running. Starting them now..."
    docker-compose -f $COMPOSE_FILE up -d
    echo "⏳ Waiting 30 seconds for containers to be ready..."
    sleep 30
fi

echo "✅ Test containers are running"
echo ""

# Step 1: Generate Prisma Client
echo "📦 Step 1: Generating Prisma Client..."
docker-compose -f $COMPOSE_FILE exec -T backend-test npx prisma generate
echo "✅ Prisma Client generated"
echo ""

# Step 2: Run migration
echo "🔄 Step 2: Running database migration..."
docker-compose -f $COMPOSE_FILE exec -T backend-test npx prisma migrate deploy
echo "✅ Migration completed"
echo ""

# Step 3: Seed default catalog
echo "🌱 Step 3: Seeding default catalog..."
docker-compose -f $COMPOSE_FILE exec -T backend-test npx ts-node prisma/seed-default-catalog.ts
echo "✅ Default catalog seeded"
echo ""

echo "========================================="
echo "✅ TEST Migration Complete!"
echo "========================================="
echo ""
echo "Test Environment Access:"
echo "  - Backend API: http://localhost:3002"
echo "  - Database: localhost:5433"
echo "  - Nginx: http://localhost:8081"
echo ""
echo "Next steps:"
echo "1. Test the multi-catalog functionality"
echo "2. Verify everything works correctly"
echo "3. When ready, deploy to production"
echo ""
echo "To view logs:"
echo "  docker-compose -f $COMPOSE_FILE logs -f backend-test"
echo ""
echo "To stop test environment:"
echo "  docker-compose -f $COMPOSE_FILE down"
echo ""
