import { useState, useEffect } from 'react';
import { useLanguage } from '../contexts/LanguageContext';
import { TrendingUp, DollarSign, FileText, ShoppingCart, Download } from 'lucide-react';
import {
    BarChart,
    Bar,
    XAxis,
    YAxis,
    CartesianGrid,
    Tooltip,
    ResponsiveContainer
} from 'recharts';

interface Invoice {
    id: string;
    number: string;
    date: string;
    customer: string;
    totalHT: number;
    totalTVA: number;
    totalTTC: number;
    status: string;
}

interface AccountingStats {
    period: {
        start: string;
        end: string;
    };
    revenue: {
        ht: number;
        ttc: number;
        currency: string;
    };
    tva: {
        total: number;
        breakdown: {
            rate19: number;
            rate7: number;
            rate0: number;
        };
        currency: string;
    };
    fiscal: {
        timbreTotal: number;
        currency: string;
    };
    profitability: {
        totalCost: number;
        grossMargin: number;
        marginPercentage: number;
        currency: string;
    };
    volume: {
        invoices: number;
        orders: number;
        averageBasketTTC: number;
    };
    charts: {
        revenue: Array<{ date: string; ht: number; ttc: number }>;
        topProducts: Array<{ name: string; revenue: number; margin: number; quantity: number }>;
    };
    invoices: Invoice[];
}

type Period = 'today' | 'thisMonth' | 'lastMonth' | 'thisYear' | 'custom';

export default function AccountingPage() {
    const { t } = useLanguage();
    const [stats, setStats] = useState<AccountingStats | null>(null);
    const [isLoading, setIsLoading] = useState(true);
    const [error, setError] = useState<string | null>(null);
    const [selectedPeriod, setSelectedPeriod] = useState<Period>('thisMonth');
    const [customDateRange] = useState({ start: '', end: '' });
    const [activeTab, setActiveTab] = useState<'dashboard' | 'audit'>('dashboard');

    useEffect(() => {
        fetchStats();
    }, [selectedPeriod, customDateRange]);

    const getDateRange = (period: Period) => {
        const now = new Date();
        let start = new Date();
        let end = new Date();

        switch (period) {
            case 'today':
                start.setHours(0, 0, 0, 0);
                end.setHours(23, 59, 59, 999);
                break;
            case 'thisMonth':
                start = new Date(now.getFullYear(), now.getMonth(), 1);
                end = new Date(now.getFullYear(), now.getMonth() + 1, 0);
                end.setHours(23, 59, 59, 999);
                break;
            case 'lastMonth':
                start = new Date(now.getFullYear(), now.getMonth() - 1, 1);
                end = new Date(now.getFullYear(), now.getMonth(), 0);
                end.setHours(23, 59, 59, 999);
                break;
            case 'thisYear':
                start = new Date(now.getFullYear(), 0, 1);
                end = new Date(now.getFullYear(), 11, 31);
                end.setHours(23, 59, 59, 999);
                break;
            case 'custom':
                if (customDateRange.start && customDateRange.end) {
                    start = new Date(customDateRange.start);
                    end = new Date(customDateRange.end);
                    end.setHours(23, 59, 59, 999);
                }
                break;
        }
        return { start: start.toISOString(), end: end.toISOString() };
    };

    const fetchStats = async () => {
        if (selectedPeriod === 'custom' && (!customDateRange.start || !customDateRange.end)) return;

        try {
            setIsLoading(true);
            const token = localStorage.getItem('admin_token');
            const { start, end } = getDateRange(selectedPeriod);

            const queryParams = new URLSearchParams({
                startDate: start,
                endDate: end
            });

            const response = await fetch(`${import.meta.env.VITE_API_URL}/admin/accounting/stats?${queryParams}`, {
                headers: {
                    'Authorization': `Bearer ${token}`,
                    'Content-Type': 'application/json',
                },
            });

            if (!response.ok) {
                throw new Error('Failed to fetch accounting stats');
            }

            const data = await response.json();
            if (data.success) {
                setStats(data.data);
            } else {
                throw new Error(data.error?.message || 'Failed to load data');
            }
        } catch (err) {
            console.error('Error fetching stats:', err);
            setError(t.common.noData);
        } finally {
            setIsLoading(false);
        }
    };

    const exportCSV = () => {
        if (!stats?.invoices) return;

        const headers = ['Invoice Number', 'Date', 'Customer', 'Total HT', 'Total TVA', 'Total TTC', 'Status'];
        const rows = stats.invoices.map(inv => [
            inv.number,
            new Date(inv.date).toLocaleDateString(),
            inv.customer,
            inv.totalHT.toFixed(3),
            inv.totalTVA.toFixed(3),
            inv.totalTTC.toFixed(3),
            inv.status
        ]);

        const csvContent = [
            headers.join(','),
            ...rows.map(row => row.join(','))
        ].join('\n');

        const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
        const link = document.createElement('a');
        const url = URL.createObjectURL(blob);
        link.setAttribute('href', url);
        link.setAttribute('download', `accounting_export_${selectedPeriod}.csv`);
        link.style.visibility = 'hidden';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    };

    if (isLoading && !stats) {
        return (
            <div className="flex justify-center items-center h-64">
                <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded relative" role="alert">
                <strong className="font-bold">Error!</strong>
                <span className="block sm:inline"> {error}</span>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header & Controls */}
            <div className="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                <div className="flex items-center gap-4">
                    <h1 className="text-2xl font-bold text-gray-900">{t.accounting.title}</h1>
                    <div className="flex bg-gray-100 rounded-lg p-1">
                        <button
                            onClick={() => setActiveTab('dashboard')}
                            className={`px-3 py-1 text-sm font-medium rounded-md transition-colors ${activeTab === 'dashboard' ? 'bg-white shadow-sm text-blue-600' : 'text-gray-600'
                                }`}
                        >
                            Dashboard
                        </button>
                        <button
                            onClick={() => setActiveTab('audit')}
                            className={`px-3 py-1 text-sm font-medium rounded-md transition-colors ${activeTab === 'audit' ? 'bg-white shadow-sm text-blue-600' : 'text-gray-600'
                                }`}
                        >
                            Audit Mode
                        </button>
                    </div>
                </div>

                <div className="flex flex-wrap items-center gap-2">
                    <div className="flex bg-white p-1 rounded-lg border border-gray-200 shadow-sm">
                        {(['today', 'thisMonth', 'lastMonth', 'thisYear'] as Period[]).map((period) => (
                            <button
                                key={period}
                                onClick={() => setSelectedPeriod(period)}
                                className={`px-3 py-1.5 text-sm font-medium rounded-md transition-colors ${selectedPeriod === period
                                    ? 'bg-blue-600 text-white shadow-sm'
                                    : 'text-gray-600 hover:bg-gray-100'
                                    }`}
                            >
                                {t.accounting[period]}
                            </button>
                        ))}
                    </div>

                    {activeTab === 'audit' && (
                        <button
                            onClick={exportCSV}
                            className="flex items-center gap-2 px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition-colors"
                        >
                            <Download className="w-4 h-4" />
                            Export CSV
                        </button>
                    )}
                </div>
            </div>

            {activeTab === 'dashboard' ? (
                <>
                    {/* Main KPIs */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                        {/* Revenue Card */}
                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="text-gray-500 text-sm font-medium">{t.accounting.revenue}</h3>
                                <div className="p-2 bg-blue-50 rounded-lg">
                                    <DollarSign className="w-5 h-5 text-blue-600" />
                                </div>
                            </div>
                            <div className="flex flex-col">
                                <div className="flex items-baseline">
                                    <span className="text-2xl font-bold text-gray-900">
                                        {stats?.revenue.ht.toFixed(3)}
                                    </span>
                                    <span className="ml-1 text-sm text-gray-500">{stats?.revenue.currency}</span>
                                </div>
                                <div className="text-xs text-gray-500 mt-1">
                                    {t.accounting.revenueTTC}: {stats?.revenue.ttc.toFixed(3)} {stats?.revenue.currency}
                                </div>
                            </div>
                        </div>

                        {/* TVA Card */}
                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="text-gray-500 text-sm font-medium">{t.accounting.tvaCollected}</h3>
                                <div className="p-2 bg-purple-50 rounded-lg">
                                    <FileText className="w-5 h-5 text-purple-600" />
                                </div>
                            </div>
                            <div className="flex flex-col">
                                <div className="flex items-baseline">
                                    <span className="text-2xl font-bold text-gray-900">
                                        {stats?.tva.total.toFixed(3)}
                                    </span>
                                    <span className="ml-1 text-sm text-gray-500">{stats?.tva.currency}</span>
                                </div>
                                <div className="text-xs text-gray-500 mt-1">
                                    {t.accounting.timbreFiscal}: {stats?.fiscal.timbreTotal.toFixed(3)} {stats?.fiscal.currency}
                                </div>
                            </div>
                        </div>

                        {/* Profit Card */}
                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="text-gray-500 text-sm font-medium">{t.accounting.grossMargin}</h3>
                                <div className="p-2 bg-green-50 rounded-lg">
                                    <TrendingUp className="w-5 h-5 text-green-600" />
                                </div>
                            </div>
                            <div className="flex flex-col">
                                <div className="flex items-baseline">
                                    <span className="text-2xl font-bold text-gray-900">
                                        {stats?.profitability.grossMargin.toFixed(3)}
                                    </span>
                                    <span className="ml-1 text-sm text-gray-500">{stats?.profitability.currency}</span>
                                </div>
                                <div className="text-xs text-gray-500 mt-1">
                                    {t.accounting.margin}: {stats?.profitability.marginPercentage.toFixed(2)}%
                                </div>
                            </div>
                        </div>

                        {/* Volume Card */}
                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="text-gray-500 text-sm font-medium">{t.accounting.volume}</h3>
                                <div className="p-2 bg-orange-50 rounded-lg">
                                    <ShoppingCart className="w-5 h-5 text-orange-600" />
                                </div>
                            </div>
                            <div className="flex flex-col">
                                <div className="flex items-baseline">
                                    <span className="text-2xl font-bold text-gray-900">
                                        {stats?.volume.invoices}
                                    </span>
                                    <span className="ml-1 text-sm text-gray-500">{t.accounting.invoices}</span>
                                </div>
                                <div className="text-xs text-gray-500 mt-1">
                                    {t.accounting.averageBasket}: {stats?.volume.averageBasketTTC.toFixed(3)} {stats?.revenue.currency}
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Charts Section */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                        {/* Revenue Evolution Chart */}
                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4">Revenue Evolution (HT)</h2>
                            <div className="h-80">
                                <ResponsiveContainer width="100%" height="100%">
                                    <BarChart data={stats?.charts.revenue}>
                                        <CartesianGrid strokeDasharray="3 3" vertical={false} />
                                        <XAxis dataKey="date" />
                                        <YAxis />
                                        <Tooltip formatter={(value: number) => `${value.toFixed(3)} TND`} />
                                        <Bar dataKey="ht" fill="#3B82F6" name="Revenue HT" radius={[4, 4, 0, 0]} />
                                    </BarChart>
                                </ResponsiveContainer>
                            </div>
                        </div>

                        {/* Top Products Table */}
                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4">Top Products</h2>
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead>
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Product</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Revenue</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Margin</th>
                                        </tr>
                                    </thead>
                                    <tbody className="bg-white divide-y divide-gray-200">
                                        {stats?.charts.topProducts.map((product, idx) => (
                                            <tr key={idx}>
                                                <td className="px-4 py-3 text-sm text-gray-900 truncate max-w-xs">{product.name}</td>
                                                <td className="px-4 py-3 text-sm text-gray-900 text-right">{product.revenue.toFixed(3)}</td>
                                                <td className="px-4 py-3 text-sm text-green-600 text-right font-medium">{product.margin.toFixed(3)}</td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    {/* TVA & Fiscal Summary */}
                    <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4">{t.accounting.tvaBreakdown}</h2>
                            <div className="space-y-4">
                                <div className="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                                    <span className="text-gray-600">TVA 19%</span>
                                    <span className="font-semibold text-gray-900">{stats?.tva.breakdown.rate19.toFixed(3)} {stats?.tva.currency}</span>
                                </div>
                                <div className="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                                    <span className="text-gray-600">TVA 7%</span>
                                    <span className="font-semibold text-gray-900">{stats?.tva.breakdown.rate7.toFixed(3)} {stats?.tva.currency}</span>
                                </div>
                                <div className="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                                    <span className="text-gray-600">TVA 0% (Exonéré)</span>
                                    <span className="font-semibold text-gray-900">{stats?.tva.breakdown.rate0.toFixed(3)} {stats?.tva.currency}</span>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white rounded-xl shadow-sm p-6 border border-gray-100">
                            <h2 className="text-lg font-semibold text-gray-900 mb-4">{t.accounting.overview}</h2>
                            <div className="space-y-4">
                                <div className="flex justify-between items-center border-b border-gray-100 pb-2">
                                    <span className="text-gray-600">{t.accounting.revenue}</span>
                                    <span className="font-medium text-gray-900">{stats?.revenue.ht.toFixed(3)} {stats?.revenue.currency}</span>
                                </div>
                                <div className="flex justify-between items-center border-b border-gray-100 pb-2">
                                    <span className="text-gray-600">{t.accounting.totalCost}</span>
                                    <span className="font-medium text-red-600">-{stats?.profitability.totalCost.toFixed(3)} {stats?.profitability.currency}</span>
                                </div>
                                <div className="flex justify-between items-center pt-2">
                                    <span className="font-bold text-gray-900">{t.accounting.grossMargin}</span>
                                    <span className="font-bold text-green-600">{stats?.profitability.grossMargin.toFixed(3)} {stats?.profitability.currency}</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </>
            ) : (
                /* Audit Mode View */
                <div className="bg-white rounded-xl shadow-sm border border-gray-100 overflow-hidden">
                    <div className="p-6 border-b border-gray-100 flex justify-between items-center">
                        <div>
                            <h2 className="text-lg font-semibold text-gray-900">Audit Journal</h2>
                            <p className="text-sm text-gray-500">Detailed list of all validated invoices for the selected period.</p>
                        </div>
                        <div className="text-right">
                            <div className="text-sm text-gray-500">Total Revenue (HT)</div>
                            <div className="text-xl font-bold text-gray-900">{stats?.revenue.ht.toFixed(3)} TND</div>
                        </div>
                    </div>
                    <div className="overflow-x-auto">
                        <table className="min-w-full divide-y divide-gray-200">
                            <thead className="bg-gray-50">
                                <tr>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Date</th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Invoice #</th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Customer</th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Total HT</th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">TVA</th>
                                    <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Total TTC</th>
                                    <th className="px-6 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                </tr>
                            </thead>
                            <tbody className="bg-white divide-y divide-gray-200">
                                {stats?.invoices.map((invoice) => (
                                    <tr key={invoice.id} className="hover:bg-gray-50">
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                            {new Date(invoice.date).toLocaleDateString()}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-blue-600">
                                            {invoice.number}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                            {invoice.customer}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900 text-right">
                                            {invoice.totalHT.toFixed(3)}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 text-right">
                                            {invoice.totalTVA.toFixed(3)}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900 text-right">
                                            {invoice.totalTTC.toFixed(3)}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-center">
                                            <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">
                                                {invoice.status}
                                            </span>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>
                </div>
            )}
        </div>
    );
}
