interface BeforeInstallPromptEvent extends Event {
  readonly platforms: string[];
  prompt: () => Promise<void>;
  userChoice: Promise<{ outcome: 'accepted' | 'dismissed'; platform: string }>;
}

type InstallAvailableListener = () => void;

let deferredPrompt: BeforeInstallPromptEvent | null = null;
const listeners = new Set<InstallAvailableListener>();
let initialized = false;

const DISMISS_KEY = 'pwa_install_dismissed';

export function initPWAInstallPrompt() {
  if (initialized || typeof window === 'undefined') {
    return;
  }

  initialized = true;

  window.addEventListener('beforeinstallprompt', (event: Event) => {
    event.preventDefault();
    deferredPrompt = event as BeforeInstallPromptEvent;
    listeners.forEach((listener) => listener());
  });

  window.addEventListener('appinstalled', () => {
    deferredPrompt = null;
  });
}

export function subscribeInstallAvailable(listener: InstallAvailableListener) {
  listeners.add(listener);
}

export function unsubscribeInstallAvailable(listener: InstallAvailableListener) {
  listeners.delete(listener);
}

export function canPromptInstall(): boolean {
  if (typeof window === 'undefined') return false;

  if (window.matchMedia && window.matchMedia('(display-mode: standalone)').matches) {
    return false;
  }

  return !!deferredPrompt;
}

export function hasDismissedInstallPrompt(): boolean {
  if (typeof window === 'undefined') return false;
  return window.localStorage.getItem(DISMISS_KEY) === '1';
}

export function setDismissedInstallPrompt(): void {
  if (typeof window === 'undefined') return;
  window.localStorage.setItem(DISMISS_KEY, '1');
}

export async function promptInstall(): Promise<'accepted' | 'dismissed' | 'unavailable'> {
  if (!deferredPrompt) {
    return 'unavailable';
  }

  const promptEvent = deferredPrompt;
  deferredPrompt = null;

  await promptEvent.prompt();

  const choice = await promptEvent.userChoice;
  if (choice.outcome === 'dismissed') {
    setDismissedInstallPrompt();
  }

  return choice.outcome;
}
