import { useCallback, useEffect, useState } from 'react';
import {
  canPromptInstall,
  hasDismissedInstallPrompt,
  promptInstall as managerPromptInstall,
  setDismissedInstallPrompt,
  subscribeInstallAvailable,
  unsubscribeInstallAvailable,
} from './installPromptManager';

type InstallOutcome = 'accepted' | 'dismissed' | 'unavailable';

interface UsePWAInstallPromptResult {
  canInstall: boolean;
  promptInstall: () => Promise<InstallOutcome>;
  dismissed: boolean;
  dismiss: () => void;
}

export function usePWAInstallPrompt(): UsePWAInstallPromptResult {
  const [canInstall, setCanInstall] = useState(false);
  const [dismissed, setDismissed] = useState(false);

  useEffect(() => {
    setCanInstall(canPromptInstall());
    setDismissed(hasDismissedInstallPrompt());

    const listener = () => {
      setCanInstall(canPromptInstall());
    };

    subscribeInstallAvailable(listener);
    return () => {
      unsubscribeInstallAvailable(listener);
    };
  }, []);

  const promptInstall = useCallback(async () => {
    const result = await managerPromptInstall();

    if (result === 'dismissed' || result === 'accepted') {
      setDismissed(true);
    }

    return result;
  }, []);

  const dismiss = useCallback(() => {
    setDismissed(true);
    setDismissedInstallPrompt();
  }, []);

  return {
    canInstall: canInstall && !dismissed,
    promptInstall,
    dismissed,
    dismiss,
  };
}
