import { useState, useEffect } from 'react';
import { ArrowLeft, FileText, Calendar, DollarSign, AlertCircle, CheckCircle, X } from 'lucide-react';
import { invoiceService, Invoice } from '../services/invoiceService';
import { useLanguage } from '../contexts/LanguageContext';

interface MyInvoicesScreenProps {
  onBack?: () => void;
}

export default function MyInvoicesScreen({ onBack }: MyInvoicesScreenProps) {
  const { t } = useLanguage();
  const [invoices, setInvoices] = useState<Invoice[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [selectedInvoice, setSelectedInvoice] = useState<Invoice | null>(null);

  useEffect(() => {
    fetchInvoices();
  }, []);

  const fetchInvoices = async () => {
    setLoading(true);
    setError('');
    try {
      const data = await invoiceService.getMyInvoices();
      setInvoices(data);
    } catch (err: any) {
      setError(err.message || t.common?.error || 'Failed to load invoices');
    } finally {
      setLoading(false);
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('fr-TN', {
      style: 'currency',
      currency: 'TND',
    }).format(amount);
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('fr-FR', {
      year: 'numeric',
      month: 'long',
      day: 'numeric',
    });
  };

  const getStatusBadge = (status: string) => {
    const statusConfig: Record<string, { bg: string; text: string; icon: any }> = {
      DRAFT: { bg: 'bg-gray-100', text: 'text-gray-700', icon: FileText },
      ISSUED: { bg: 'bg-blue-100', text: 'text-blue-700', icon: AlertCircle },
      PAID: { bg: 'bg-green-100', text: 'text-green-700', icon: CheckCircle },
      OVERDUE: { bg: 'bg-red-100', text: 'text-red-700', icon: AlertCircle },
      CANCELLED: { bg: 'bg-gray-100', text: 'text-gray-700', icon: AlertCircle },
    };

    const config = statusConfig[status] || statusConfig.ISSUED;
    const Icon = config.icon;

    let statusLabel = status;
    if (status === 'DRAFT') statusLabel = t.invoices?.draft || status;
    if (status === 'ISSUED') statusLabel = t.invoices?.issued || status;
    if (status === 'PAID') statusLabel = t.invoices?.paid || status;
    if (status === 'OVERDUE') statusLabel = t.invoices?.overdue || status;
    if (status === 'CANCELLED') statusLabel = t.invoices?.cancelled || status;

    return (
      <span className={`inline-flex items-center gap-1 px-3 py-1 rounded-full text-sm font-medium ${config.bg} ${config.text}`}>
        <Icon className="w-4 h-4" />
        {statusLabel}
      </span>
    );
  };

  const isOverdue = (invoice: Invoice) => {
    if (!invoice.dueDate || invoice.status === 'PAID') return false;
    return new Date(invoice.dueDate) < new Date();
  };

  return (
    <div className="min-h-screen bg-gray-50 pb-20">
      {/* Header */}
      <div className="bg-white border-b border-gray-200 sticky top-0 z-10">
        <div className="max-w-7xl mx-auto px-4 py-4">
          <div className="flex items-center gap-4">
            {onBack && (
              <button
                onClick={onBack}
                className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
              >
                <ArrowLeft className="w-6 h-6 text-gray-600" />
              </button>
            )}
            <div>
              <h1 className="text-2xl font-bold text-gray-900">{t.invoices?.title || 'My Invoices'}</h1>
              <p className="text-sm text-gray-600">{t.invoices?.subtitle || 'View and manage your invoices'}</p>
            </div>
          </div>
        </div>
      </div>

      <div className="max-w-7xl mx-auto px-4 py-6">
        {/* Error Message */}
        {error && (
          <div className="bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg mb-6">
            {error}
          </div>
        )}

        {/* Loading State */}
        {loading ? (
          <div className="bg-white rounded-lg shadow-sm p-12 text-center">
            <div className="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
            <p className="mt-4 text-gray-600">{t.common?.loading || 'Loading...'}</p>
          </div>
        ) : invoices.length === 0 ? (
          /* Empty State */
          <div className="bg-white rounded-lg shadow-sm p-12 text-center">
            <FileText className="w-16 h-16 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-semibold text-gray-900 mb-2">{t.invoices?.noInvoices || 'No Invoices Yet'}</h3>
            <p className="text-gray-600 mb-6">
              {t.invoices?.noInvoicesMessage || 'Invoices will appear here once your orders are shipped or delivered.'}
            </p>
            {onBack && (
              <button
                onClick={onBack}
                className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
              >
                {t.invoices?.backToAccount || 'Back to Account'}
              </button>
            )}
          </div>
        ) : (
          /* Invoices List */
          <div className="space-y-4">
            {invoices.map((invoice) => (
              <div
                key={invoice.id}
                className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover:shadow-md transition-shadow cursor-pointer"
                onClick={() => setSelectedInvoice(invoice)}
              >
                <div className="flex items-start justify-between mb-4">
                  <div>
                    <h3 className="text-lg font-bold text-gray-900">{invoice.invoiceNumber}</h3>
                    <p className="text-sm text-gray-600">
                      {t.invoices?.issuedDate || 'Issued'}: {formatDate(invoice.issuedDate)}
                    </p>
                  </div>
                  {getStatusBadge(invoice.status)}
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                  <div className="flex items-center gap-2">
                    <DollarSign className="w-5 h-5 text-gray-400" />
                    <div>
                      <p className="text-xs text-gray-600">{t.invoices?.total || 'Total Amount'}</p>
                      <p className="font-semibold text-gray-900">{formatCurrency(invoice.totalAmount)}</p>
                    </div>
                  </div>

                  {invoice.dueDate && (
                    <div className="flex items-center gap-2">
                      <Calendar className="w-5 h-5 text-gray-400" />
                      <div>
                        <p className="text-xs text-gray-600">{t.invoices?.dueDate || 'Due Date'}</p>
                        <p className={`font-semibold ${isOverdue(invoice) ? 'text-red-600' : 'text-gray-900'}`}>
                          {formatDate(invoice.dueDate)}
                        </p>
                      </div>
                    </div>
                  )}

                  {invoice.paymentTerm && (
                    <div className="flex items-center gap-2">
                      <FileText className="w-5 h-5 text-gray-400" />
                      <div>
                        <p className="text-xs text-gray-600">{t.invoices?.paymentTerms || 'Payment Terms'}</p>
                        <p className="font-semibold text-gray-900">{invoice.paymentTerm.replace('_', ' ')}</p>
                      </div>
                    </div>
                  )}
                </div>

                {isOverdue(invoice) && (
                  <div className="bg-red-50 border border-red-200 rounded-lg p-3 flex items-center gap-2">
                    <AlertCircle className="w-5 h-5 text-red-600 flex-shrink-0" />
                    <p className="text-sm text-red-700 font-medium">{t.invoices?.overdueMessage || 'This invoice is overdue'}</p>
                  </div>
                )}
              </div>
            ))}
          </div>
        )}
      </div>

      {/* Invoice Detail Modal */}
      {selectedInvoice && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white rounded-lg max-w-4xl w-full max-h-[90vh] overflow-y-auto">
            {/* Modal Header */}
            <div className="sticky top-0 bg-white border-b border-gray-200 px-6 py-4 flex items-center justify-between">
              <div>
                <h2 className="text-2xl font-bold text-gray-900">{t.invoices?.invoiceDetails || 'Invoice Details'}</h2>
                <p className="text-sm text-gray-600 mt-1">{selectedInvoice.invoiceNumber}</p>
              </div>
              <button
                onClick={() => setSelectedInvoice(null)}
                className="p-2 hover:bg-gray-100 rounded-lg transition-colors"
              >
                <X className="w-6 h-6 text-gray-600" />
              </button>
            </div>

            {/* Modal Content */}
            <div className="p-6">
              {/* Invoice Header Info */}
              <div className="flex justify-between mb-6">
                <div className="space-y-2">
                  <div>
                    <span className="text-sm text-gray-600">{t.invoices?.invoiceNumber || 'Number'}:</span>
                    <span className="ml-2 font-semibold text-gray-900">{selectedInvoice.invoiceNumber}</span>
                  </div>
                  <div>
                    <span className="text-sm text-gray-600">{t.invoices?.issuedDate || 'Date'}:</span>
                    <span className="ml-2 font-medium text-gray-900">{formatDate(selectedInvoice.issuedDate)}</span>
                  </div>
                </div>
                <div className="text-right">
                  <p className="text-sm font-semibold text-gray-900">{t.invoices?.billingContact || 'Billing Contact'}:</p>
                  {selectedInvoice.user?.companyName && (
                    <p className="text-sm text-gray-700">{selectedInvoice.user.companyName}</p>
                  )}
                  <p className="text-sm text-gray-700">
                    {selectedInvoice.user?.firstName} {selectedInvoice.user?.lastName}
                  </p>
                  <p className="text-sm text-gray-600">{selectedInvoice.user?.email}</p>
                </div>
              </div>

              {/* Items Table */}
              {selectedInvoice.items && selectedInvoice.items.length > 0 ? (
                <div className="mb-6">
                  <div className="bg-cyan-50 border border-cyan-200 rounded-t-lg overflow-hidden">
                    <table className="w-full">
                      <thead>
                        <tr className="bg-cyan-100">
                          <th className="px-4 py-3 text-left text-sm font-semibold text-gray-700">{t.invoices?.product || 'Product'}</th>
                          <th className="px-4 py-3 text-center text-sm font-semibold text-gray-700">{t.invoices?.unitPrice || 'Unit Price'}</th>
                          <th className="px-4 py-3 text-center text-sm font-semibold text-gray-700">{t.invoices?.quantity || 'Quantity'}</th>
                          <th className="px-4 py-3 text-right text-sm font-semibold text-gray-700">{t.invoices?.priceHT || 'Total HT'}</th>
                        </tr>
                      </thead>
                      <tbody className="bg-white">
                        {selectedInvoice.items.map((item, index) => (
                          <tr key={item.id} className={index % 2 === 0 ? 'bg-white' : 'bg-gray-50'}>
                            <td className="px-4 py-3">
                              <div>
                                <p className="font-medium text-gray-900">{item.productName}</p>
                                {item.productSku && (
                                  <p className="text-xs text-gray-500">{item.productSku}</p>
                                )}
                              </div>
                            </td>
                            <td className="px-4 py-3 text-center text-sm text-gray-700">
                              {formatCurrency(item.unitPrice)}
                            </td>
                            <td className="px-4 py-3 text-center text-sm text-gray-700">
                              {item.quantity}
                            </td>
                            <td className="px-4 py-3 text-right font-medium text-gray-900">
                              {formatCurrency(item.totalHT)}
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>

                  {/* Subtotal */}
                  <div className="bg-cyan-50 border-x border-b border-cyan-200 px-4 py-2 flex justify-between rounded-b-lg">
                    <span className="font-semibold text-gray-700">{t.invoices?.subtotal || 'SUBTOTAL'}</span>
                    <span className="font-semibold text-gray-900">
                      {formatCurrency(selectedInvoice.totalNetHT || selectedInvoice.subtotal)}
                    </span>
                  </div>
                </div>
              ) : (
                <div className="mb-6 p-4 bg-gray-50 border border-gray-200 rounded-lg text-center text-gray-600">
                  {t.invoices?.noItems || 'No items available'}
                </div>
              )}

              {/* Financial Summary */}
              <div className="flex justify-end mb-6">
                <div className="w-full max-w-sm space-y-2">
                  {/* Total HT */}
                  <div className="flex justify-between items-center px-4 py-2 bg-cyan-50 rounded">
                    <span className="text-sm font-medium text-gray-700">{t.invoices?.totalHT || 'Total HT'}</span>
                    <span className="font-semibold text-gray-900">
                      {formatCurrency(selectedInvoice.totalNetHT || selectedInvoice.subtotal)}
                    </span>
                  </div>

                  {/* TVA Breakdown */}
                  {selectedInvoice.tva19Amount !== undefined && selectedInvoice.tva19Amount > 0 && (
                    <div className="flex justify-between items-center px-4 py-2">
                      <span className="text-sm text-gray-600">{t.invoices?.tva19 || 'TVA 19%'}</span>
                      <span className="text-sm text-gray-900">{formatCurrency(selectedInvoice.tva19Amount)}</span>
                    </div>
                  )}
                  {selectedInvoice.tva7Amount !== undefined && selectedInvoice.tva7Amount > 0 && (
                    <div className="flex justify-between items-center px-4 py-2">
                      <span className="text-sm text-gray-600">{t.invoices?.tva7 || 'TVA 7%'}</span>
                      <span className="text-sm text-gray-900">{formatCurrency(selectedInvoice.tva7Amount)}</span>
                    </div>
                  )}
                  {selectedInvoice.tva0Amount !== undefined && selectedInvoice.tva0Amount > 0 && (
                    <div className="flex justify-between items-center px-4 py-2">
                      <span className="text-sm text-gray-600">{t.invoices?.tva0 || 'TVA 0%'}</span>
                      <span className="text-sm text-gray-900">{formatCurrency(selectedInvoice.tva0Amount)}</span>
                    </div>
                  )}

                  {/* Total TVA */}
                  {!selectedInvoice.totalTVA && selectedInvoice.taxAmount > 0 && (
                    <div className="flex justify-between items-center px-4 py-2">
                      <span className="text-sm text-gray-600">{t.invoices?.tva || 'TVA'}</span>
                      <span className="text-sm text-gray-900">{formatCurrency(selectedInvoice.taxAmount)}</span>
                    </div>
                  )}

                  {/* Total TTC */}
                  <div className="flex justify-between items-center px-4 py-3 bg-blue-900 text-white rounded font-bold">
                    <span className="text-lg">{t.invoices?.totalTTC || 'TOTAL TTC'}</span>
                    <span className="text-xl">
                      {formatCurrency(selectedInvoice.netAPayer || selectedInvoice.totalAmount)}
                    </span>
                  </div>
                </div>
              </div>

              {/* Payment Info */}
              <div className="grid grid-cols-2 gap-4 p-4 bg-gray-50 rounded-lg">
                <div>
                  <p className="text-sm text-gray-600">{t.invoices?.status || 'Status'}</p>
                  <div className="mt-1">{getStatusBadge(selectedInvoice.status)}</div>
                </div>
                {selectedInvoice.paymentTerm && (
                  <div>
                    <p className="text-sm text-gray-600">{t.invoices?.paymentTerms || 'Payment Term'}</p>
                    <p className="mt-1 font-medium text-gray-900">{selectedInvoice.paymentTerm}</p>
                  </div>
                )}
                {selectedInvoice.dueDate && (
                  <div>
                    <p className="text-sm text-gray-600">{t.invoices?.dueDate || 'Due Date'}</p>
                    <p className="mt-1 font-medium text-gray-900">{formatDate(selectedInvoice.dueDate)}</p>
                  </div>
                )}
                {selectedInvoice.paidDate && (
                  <div>
                    <p className="text-sm text-gray-600">{t.invoices?.paidOn || 'Paid Date'}</p>
                    <p className="mt-1 font-medium text-gray-900">{formatDate(selectedInvoice.paidDate)}</p>
                  </div>
                )}
              </div>

              {/* Notes */}
              {selectedInvoice.notes && (
                <div className="mt-4 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                  <p className="text-sm font-semibold text-gray-700 mb-1">Notes</p>
                  <p className="text-sm text-gray-600">{selectedInvoice.notes}</p>
                </div>
              )}
            </div>

            {/* Modal Footer */}
            <div className="sticky bottom-0 bg-gray-50 px-6 py-4 border-t border-gray-200 flex justify-end gap-3">
              <button
                onClick={() => setSelectedInvoice(null)}
                className="px-6 py-2 border border-gray-300 bg-white text-gray-700 rounded-lg hover:bg-gray-50 font-medium transition-colors"
              >
                {t.common?.close || 'Close'}
              </button>
              <button
                onClick={() => window.print()}
                className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 font-medium transition-colors"
              >
                {t.invoices?.printInvoice || 'Print Invoice'}
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}
