// Cart debugging utilities

export const CART_STORAGE_KEY = 'hardware-store-cart';

// Clear cart from localStorage
export const clearCartStorage = () => {
  try {
    localStorage.removeItem(CART_STORAGE_KEY);
    console.log('Cart cleared from localStorage');
  } catch (error) {
    console.error('Error clearing cart from localStorage:', error);
  }
};

// Get current cart from localStorage
export const getCartFromStorage = () => {
  try {
    const savedCart = localStorage.getItem(CART_STORAGE_KEY);
    return savedCart ? JSON.parse(savedCart) : [];
  } catch (error) {
    console.error('Error getting cart from localStorage:', error);
    return [];
  }
};

// Log current cart contents
export const logCartContents = () => {
  const cart = getCartFromStorage();
  console.log('Current cart contents:', cart);
  console.log('Number of items:', cart.length);
  if (cart.length > 0) {
    cart.forEach((item: any, index: number) => {
      console.log(`Item ${index + 1}:`, {
        productId: item.product?.id,
        productName: item.product?.name,
        quantity: item.quantity,
        cartItemKey: item.cartItemKey
      });
    });
  }
};

// Check if cart has problematic data
export const validateCartData = (): string[] => {
  const cart = getCartFromStorage();
  const issues: string[] = [];
  
  if (!Array.isArray(cart)) {
    issues.push('Cart is not an array');
    return issues;
  }
  
  cart.forEach((item: any, index: number) => {
    if (!item.product || !item.product.id) {
      issues.push(`Item ${index}: Missing product data`);
    }
    if (!item.cartItemKey) {
      issues.push(`Item ${index}: Missing cartItemKey`);
    }
    if (typeof item.quantity !== 'number' || item.quantity <= 0) {
      issues.push(`Item ${index}: Invalid quantity`);
    }
  });
  
  return issues;
};
