import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function deleteTestAccounts() {
    console.log('=== DELETING TEST ACCOUNTS ===\n');

    const emailsToDelete = [
        'smpppppd@tunisie.com',
        'damisofyan@gmail.com',
        'abc@tunisie.com',
        'abc@gmail.com',
        'other@gmail.com',
    ];

    for (const email of emailsToDelete) {
        try {
            const user = await prisma.user.findUnique({
                where: { email },
                include: {
                    catalogAssignments: true,
                    orders: true,
                    quotations: true,
                }
            });

            if (!user) {
                console.log(`⚠️  User ${email} not found in database`);
                continue;
            }

            // Delete related data first
            if (user.catalogAssignments.length > 0) {
                await prisma.catalogCustomer.deleteMany({
                    where: { customerId: user.id }
                });
                console.log(`  - Deleted ${user.catalogAssignments.length} catalog assignments`);
            }

            if (user.orders.length > 0) {
                // Delete order items first
                for (const order of user.orders) {
                    await prisma.orderItem.deleteMany({
                        where: { orderId: order.id }
                    });
                    await prisma.orderStatusHistory.deleteMany({
                        where: { orderId: order.id }
                    });
                }
                await prisma.order.deleteMany({
                    where: { userId: user.id }
                });
                console.log(`  - Deleted ${user.orders.length} orders`);
            }

            if (user.quotations.length > 0) {
                // Delete quotation items first
                for (const quotation of user.quotations) {
                    await prisma.quotationItem.deleteMany({
                        where: { quotationId: quotation.id }
                    });
                }
                await prisma.quotation.deleteMany({
                    where: { userId: user.id }
                });
                console.log(`  - Deleted ${user.quotations.length} quotations`);
            }

            // Delete refresh tokens
            await prisma.refreshToken.deleteMany({
                where: { userId: user.id }
            });

            // Finally delete the user
            await prisma.user.delete({
                where: { id: user.id }
            });

            console.log(`✅ Deleted user: ${email}\n`);
        } catch (error) {
            console.error(`❌ Error deleting ${email}:`, error);
        }
    }

    console.log('=== CLEANUP COMPLETE ===');
}

deleteTestAccounts()
    .catch(console.error)
    .finally(() => prisma.$disconnect());
