// Node 20+ has native fetch

const API_URL = 'http://localhost:3001/api';

async function reproduceIssue() {
    console.log('🔄 Starting Quotation Approval Reproduction...');

    let adminToken = '';
    let customerToken = '';
    let quotationId = '';

    // 1. Login as Admin
    try {
        const res = await fetch(`${API_URL}/auth/login`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ email: 'admin@smd-tunisie.com', password: 'admin123' })
        });
        const data = await res.json() as any;
        if (res.ok) {
            adminToken = data.data.token;
            console.log('✅ Admin Login Success');
        } else {
            console.error('❌ Admin Login Failed:', data);
            return;
        }
    } catch (e) {
        console.error('❌ API Unreachable:', e);
        return;
    }

    // 2. Create and/or Login Customer
    // specific customer for testing
    const customerEmail = `test-cust-${Date.now()}@test.com`;
    const customerPassword = 'password123';

    try {
        // Register customer
        // Note: Public registration might be disabled, so we might need to use admin to create customer or use an existing one.
        // Let's try to register first. If it fails (due to B2B enforcement), we use admin to create one.
        // Actually, B2B enforcement disabled public register.
        // So let's use Admin to create a customer.

        const createCustRes = await fetch(`${API_URL}/customers`, { // Correct endpoint
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminToken}`
            },
            body: JSON.stringify({
                email: customerEmail,
                password: customerPassword,
                firstName: 'Test',
                lastName: 'Customer',
                phone: '22222222',
                companyName: 'Test Company',
                rneNumber: `RNE${Date.now()}`,
                status: 'ACTIVE'
            })
        });

        if (!createCustRes.ok && createCustRes.status !== 404) {
            // If 404, maybe endpoint is different. Try /auth/register just in case it is open for now
            console.log('⚠️ Failed to create customer via Admin API, trying public register...');
            console.log('Error details:', await createCustRes.text());
            const registerRes = await fetch(`${API_URL}/auth/register`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({
                    email: customerEmail,
                    password: customerPassword,
                    firstName: 'Test',
                    lastName: 'Customer',
                    phone: '22222222'
                })
            });

            if (!registerRes.ok) {
                console.error('❌ Failed to register customer:', await registerRes.text());
                return;
            }
        } else if (createCustRes.ok) {
            console.log('✅ Customer Created via Admin');
        }

        // Login Customer
        const loginRes = await fetch(`${API_URL}/auth/login`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ email: customerEmail, password: customerPassword })
        });
        const loginData = await loginRes.json() as any;
        customerToken = loginData.data?.token;
        if (!customerToken) {
            console.error('❌ Customer Login Failed:', loginData);
            return;
        }
        console.log('✅ Customer Login Success');

    } catch (e) {
        console.error('❌ Customer Setup Failed:', e);
        return;
    }

    // 3. Create Quotation as Customer
    try {
        // We need a product ID. Fetch products first.
        const productsRes = await fetch(`${API_URL}/products?page=1&limit=1`, {
            headers: { 'Authorization': `Bearer ${customerToken}` } // Depending on isolation, might need specific headers
        });
        const productsData = await productsRes.json() as any;
        const product = productsData.data?.products?.[0];

        if (!product) {
            console.error('❌ No products found to create quotation');
            return;
        }

        const createQuotRes = await fetch(`${API_URL}/quotations`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${customerToken}`
            },
            body: JSON.stringify({
                items: [
                    { productId: product.id, quantity: 1 }
                ]
            })
        });
        const quotData = await createQuotRes.json() as any;
        quotationId = quotData.data?.id;

        if (createQuotRes.ok && quotationId) {
            console.log(`✅ Quotation Created: ${quotationId}`);
        } else {
            console.error('❌ Create Quotation Failed:', quotData);
            return;
        }

    } catch (e) {
        console.error('❌ Quotation Creation Failed:', e);
        return;
    }

    // 4. Verify Visibility as Admin
    try {
        console.log(`👉 Checking visibility of quotation ${quotationId} as Admin...`);
        const listRes = await fetch(`${API_URL}/quotations?page=1&limit=50`, {
            headers: {
                'Authorization': `Bearer ${adminToken}`
            }
        });
        const listData = await listRes.json() as any;
        const found = listData.data?.some((q: any) => q.id === quotationId);

        if (found) {
            console.log('✅ Quotation is VISIBLE to Admin.');
        } else {
            console.error('❌ Quotation is NOT visible to Admin in the list!');
            return;
        }

    } catch (e) {
        console.error('❌ Visibility Check Failed:', e);
        return;
    }

    // 5. Approve Quotation as Admin
    try {
        console.log(`👉 Attempting to approve quotation ${quotationId} as Admin...`);
        const approveRes = await fetch(`${API_URL}/quotations/${quotationId}/approve`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
                'Authorization': `Bearer ${adminToken}`
            }
        });

        if (approveRes.ok) {
            console.log('✅ Quotation Approved Successfully! Fix Verified.');
        } else {
            console.error('❌ Quotation Approval Failed:', await approveRes.text());
        }

    } catch (e) {
        console.error('❌ Approval Request Failed:', e);
    }
}

reproduceIssue();
