import { useState, useEffect } from 'react';
import { Save, Check, Calculator } from 'lucide-react';
import { buildAuthHeaders } from '../utils/apiClient';

interface TaxConfig {
    standardRate: number;
    reducedRate: number;
    fodecRate: number;
    enableStamp: boolean;
    stampValue: number;
}

export default function TaxSettingsPage() {
    const [config, setConfig] = useState<TaxConfig>({
        standardRate: 19,
        reducedRate: 7,
        fodecRate: 1,
        enableStamp: true,
        stampValue: 1.000,
    });
    const [loading, setLoading] = useState(false);
    const [saved, setSaved] = useState(false);
    const [error, setError] = useState<string | null>(null);

    const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

    useEffect(() => {
        let mounted = true;
        const fetchConfig = async () => {
            try {
                const token = localStorage.getItem('admin_token');
                if (!token) return;

                const res = await fetch(`${API_URL}/admin/plugins/tax-tunisia/config`, {
                    headers: buildAuthHeaders()
                });

                if (res.ok) {
                    const data = await res.json();
                    if (mounted && data.success && data.data && Object.keys(data.data).length > 0) {
                        setConfig(prev => ({ ...prev, ...data.data }));
                    }
                }
            } catch (err) {
                console.error('Failed to load tax config', err);
            }
        };
        fetchConfig();
        return () => { mounted = false; };
    }, []);

    const handleSave = async () => {
        setLoading(true);
        setError(null);
        try {
            const res = await fetch(`${API_URL}/admin/plugins/tax-tunisia/config`, {
                method: 'PUT',
                headers: buildAuthHeaders(),
                body: JSON.stringify(config)
            });

            if (!res.ok) throw new Error('Failed to save');

            setSaved(true);
            setTimeout(() => setSaved(false), 3000);
        } catch (err) {
            setError('Failed to save configuration');
        } finally {
            setLoading(false);
        }
    };

    return (
        <div className="space-y-6">
            {error && <div className="p-4 bg-red-50 text-red-600 rounded-lg">{error}</div>}
            <div>
                <h1 className="text-2xl font-bold text-gray-900">Tax Configuration (Tunisia)</h1>
                <p className="text-gray-600 mt-1">Manage VAT rates, FODEC, and Timber Stamp settings.</p>
            </div>

            <div className="bg-white rounded-xl border border-gray-200 p-6">
                <div className="flex items-center gap-4 mb-6 p-4 bg-blue-50 text-blue-800 rounded-lg border border-blue-100">
                    <Calculator className="w-6 h-6" />
                    <div>
                        <h3 className="font-semibold">Plugin Active: tax-tunisia v1.0.0</h3>
                        <p className="text-sm">These settings control the global tax calculation logic provided by the active plugin.</p>
                    </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                    {/* VAT Rates */}
                    <div className="space-y-4">
                        <h3 className="text-lg font-medium text-gray-900">Value Added Tax (TVA)</h3>

                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">Standard Rate (%)</label>
                            <input
                                type="number"
                                value={config.standardRate}
                                onChange={e => setConfig({ ...config, standardRate: parseFloat(e.target.value) })}
                                className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                            />
                            <p className="text-xs text-gray-500 mt-1">Default rate for most products (usually 19%)</p>
                        </div>

                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">Reduced Rate (%)</label>
                            <input
                                type="number"
                                value={config.reducedRate}
                                onChange={e => setConfig({ ...config, reducedRate: parseFloat(e.target.value) })}
                                className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                            />
                            <p className="text-xs text-gray-500 mt-1">Rate for specific categories (usually 7%)</p>
                        </div>
                    </div>

                    {/* Other Taxes */}
                    <div className="space-y-4">
                        <h3 className="text-lg font-medium text-gray-900">Additional Levies</h3>

                        <div>
                            <label className="block text-sm font-medium text-gray-700 mb-1">FODEC Rate (%)</label>
                            <input
                                type="number"
                                value={config.fodecRate}
                                onChange={e => setConfig({ ...config, fodecRate: parseFloat(e.target.value) })}
                                className="w-full px-4 py-2 border border-gray-300 rounded-lg"
                            />
                            <p className="text-xs text-gray-500 mt-1">Development fund tax (usually 1%)</p>
                        </div>

                        <div className="pt-2">
                            <label className="flex items-center gap-2 cursor-pointer mb-2">
                                <input
                                    type="checkbox"
                                    checked={config.enableStamp}
                                    onChange={e => setConfig({ ...config, enableStamp: e.target.checked })}
                                    className="w-4 h-4 text-blue-600 rounded"
                                />
                                <span className="font-medium text-gray-700">Enable Timber Stamp (Timbre Fiscal)</span>
                            </label>

                            {config.enableStamp && (
                                <div>
                                    <input
                                        type="number"
                                        value={config.stampValue}
                                        onChange={e => setConfig({ ...config, stampValue: parseFloat(e.target.value) })}
                                        step="0.100"
                                        className="w-full max-w-[200px] px-4 py-2 border border-gray-300 rounded-lg"
                                    />
                                    <p className="text-xs text-gray-500 mt-1">Fixed amount per invoice (usually 1.000 TND)</p>
                                </div>
                            )}
                        </div>
                    </div>
                </div>

                <div className="mt-8 pt-6 border-t border-gray-100 flex justify-end gap-3">
                    {saved && (
                        <span className="flex items-center text-green-600 gap-1 animate-fade-in">
                            <Check className="w-4 h-4" /> Saved
                        </span>
                    )}
                    <button
                        onClick={handleSave}
                        disabled={loading}
                        className="flex items-center gap-2 px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50"
                    >
                        <Save className="w-4 h-4" />
                        Save Changes
                    </button>
                </div>
            </div>
        </div>
    );
}
