import React from 'react';

// Type for a React Component that can be plugged in
type PlugComponent = React.FC<any>;

class ComponentRegistry {
    private slots: Map<string, PlugComponent[]> = new Map();

    /**
     * Register a component to a specific slot name
     * @param slotName e.g., 'product_page_actions'
     * @param component The React component to render
     */
    register(slotName: string, component: PlugComponent) {
        if (!this.slots.has(slotName)) {
            this.slots.set(slotName, []);
        }
        this.slots.get(slotName)?.push(component);
        console.log(`[ComponentRegistry] Registered component to slot '${slotName}'`);
    }

    /**
     * Get all components registered to a slot
     */
    get(slotName: string): PlugComponent[] {
        return this.slots.get(slotName) || [];
    }
}

export const registry = new ComponentRegistry();

/**
 * Helper Component to render a slot
 */
interface ExtensionSlotProps {
    name: string;
    [key: string]: any; // Props to pass to the plugged components
}

export const ExtensionSlot: React.FC<ExtensionSlotProps> = ({ name, ...props }) => {
    const components = registry.get(name);

    if (components.length === 0) return null;

    return (
        <>
        {
            components.map((Component, index) => (
                <Component key= {`${name}-${index}`} {...props } />
      ))}
</>
  );
};
