
# 🔌 Plugin Developer Guide

Welcome to the **SQB Hardware Store** plugin ecosystem. This guide outlines the **standard operating procedures**, **best practices**, and **architectural principles** for developing high-quality plugins for the system.

---

## 🏗️ Architectural Philosophy

The system uses a **Microkernel Architecture**:
*   **The Core**: Minimal, stable, and unaware of specific business logic nuances (e.g., specific country tax rules).
*   **The Plugins**: Encapsulate volatile or specific business logic (e.g., `tax-tunisia`, `inventory-management`).

### Core Systems
1.  **HookSystem (Backend)**: A pipeline/waterfall event bus. Plugins "listen" to events (`calculate.line_item`) and modify the data flow.
2.  **ComponentRegistry (Frontend)**: A "Slot" system. The core UI defines slots (e.g., `product_actions`), and plugins "inject" React components into them.

---

## ✅ The "Dos" (Best Practices)

### 1. Do Use the Hook System
**Why?** It ensures your logic upgrades safely without modifying core files.
*   **Correct**: Listen to `calculate.document` to add a footer note.
*   **Incorrect**: Modifying `OrderController.ts` directly to add a footer note.

### 2. Do Handle Errors Gracefully
**Why?** One bad plugin should not crash the entire server.
*   **Practice**: Wrap your hook logic in `try/catch`. If your calculation fails, return the original input or a safe fallback.
*   **Code Example**:
    ```typescript
    hooks.on('calculate.line_item', async (ctx, next) => {
        try {
            // ... risky logic ...
        } catch (e) {
            console.error('[MyPlugin] Calculation failed, falling back to default', e);
        }
        await next(); // ALWAYS call next()
    });
    ```

### 3. Do Use Dynamic Configuration
**Why?** Hardcoded values require code changes to update.
*   **Practice**: Use `PluginManager.getPluginConfig('my-plugin')` to retrieve settings.
*   **Feature**: exposes an invalidable configuration endpoint so Admin users can change settings (like Tax Rates) without a server restart/redeploy.

### 4. Do Namespace Your Data
**Why?** To prevent collisions with other plugins.
*   **Practice**: If you store data in a generic field (like `metadata` in JSON), prefix your keys.
    *   `metadata: { "myPlugin_retryCount": 5 }`

---

## ❌ The "Don'ts" (Anti-Patterns)

### 1. Don't Block the Event Loop
**Why?** Node.js is single-threaded. Heavy sync operations will freeze the server for everyone.
*   **Avoid**: Synchronous file I/O (`fs.readFileSync`), heavy CPU loops.
*   **Use**: Asynchronous APIs (`fs.promises`), offload heavy compute to worker threads if necessary.

### 2. Don't Bypass Core Security
**Why?** Plugins run with full privileges.
*   **Avoid**: Creating API routes that skip `AuthMiddleware` unless absolutely necessary public endpoints (e.g., webhooks).
*   **Check**: Always verify `req.user.role` if you expose new administrative endpoints.

### 3. Don't Modify Core Database Schema Directly (Unless Managed)
**Why?** Prisma schema is the source of truth.
*   **Avoid**: Manually running `ALTER TABLE` SQL commands that conflict with Prisma migrations.
*   **Preferred**: Store extra data in JSON fields (`metadata`), or if you must extend the schema, submit a core Pull Request.

### 4. Don't Hardcode Frontend Slots
**Why?** If the core layout changes, your hardcoded selector will break.
*   **Avoid**: Using `document.querySelector('#some-div')` to inject UI.
*   **Use**: `registry.register('slot_name', MyComponent)` and rely on the core `<ExtensionSlot />`.

---

## 🛠️ Development Workflow

### Backend Plugin Development
1.  **Draft**: Create logic in a separate folder (`src/plugins/my-plugin`).
2.  **Test**: Write a unit test using Jest to mock the `HookSystem` context.
3.  **Integrate**: Import and call `init(app, hooks)` in `server.ts` (for local dev).
4.  **Package**: When ready for production, remove the manual import and **Zip** the folder for dynamic upload via the Admin Dashboard.

### Frontend Plugin Development
1.  **Component**: Build your React component in isolation.
2.  ** Register**: Add to `src/plugins/my-plugin/index.ts`.
3.  **Test**: Run `npm run dev` and check the UI slot.
4.  **Deploy**: Frontend plugins are **static**. You must commit the code and rebuild/redeploy the frontend application.

---

## 🔒 Security Checklist

- [ ] **Input Validation**: Do your API endpoints validate `req.body`? (Use Zod or Joi).
- [ ] **Sanitization**: Are you escaping user input before rendering it or using it in raw SQL?
- [ ] **Permissions**: Do your navigation items restrict visibility based on roles?
    ```typescript
    roles: ['ADMIN'] // Good
    roles: []        // Visible to everyone? Be careful.
    ```

---

## 🚀 Performance Tips

*   **Caching**: If your plugin fetches external data (e.g., currency rates), cache the result for a few minutes. Don't hit the external API on every request.
*   **Lazy Loading**: For frontend plugins, use `React.lazy()` if the component is large and not immediately visible.

---

**Happy Coding!** 🚀
*The system is designed to be as extensible as you make it.*
