import { ReactNode } from 'react';
import { Link, useLocation } from 'react-router-dom';
import {
  LayoutDashboard,
  Package,
  FolderTree,
  ShoppingCart,
  Users,
  Settings,
  LogOut,
  Menu,
  X,
  Megaphone,
  FileText,
  Blocks
} from 'lucide-react';
import { useState, useEffect } from 'react';
import { useLanguage } from '../contexts/LanguageContext';
import CatalogSelector from './CatalogSelector';
import { useAuth, UserRole } from '../contexts/AuthContext';

interface DashboardLayoutProps {
  children: ReactNode;
  onLogout: () => void;
}

export default function DashboardLayout({ children, onLogout }: DashboardLayoutProps) {
  const location = useLocation();
  const [sidebarOpen, setSidebarOpen] = useState(false);
  const { t } = useLanguage();
  const { hasRole } = useAuth(); // ADDED

  const navigation = [
    { name: t.nav.dashboard, href: '/', icon: LayoutDashboard, roles: [] }, // All
    { name: t.nav.products, href: '/products', icon: Package, roles: [UserRole.CATALOG_ADMIN] },
    { name: t.nav.categories, href: '/categories', icon: FolderTree, roles: [UserRole.CATALOG_ADMIN] },
    { name: t.nav.catalogs, href: '/catalogs', icon: FolderTree, roles: [UserRole.CATALOG_ADMIN] },
    { name: t.nav.orders, href: '/orders', icon: ShoppingCart, roles: [UserRole.SALES_ADMIN, UserRole.SUPPORT_ADMIN] },
    { name: t.nav.quotations, href: '/quotations', icon: FileText, roles: [UserRole.SALES_ADMIN, UserRole.SUPPORT_ADMIN] },
    { name: t.nav.customers, href: '/customers', icon: Users, roles: [UserRole.SALES_ADMIN, UserRole.SUPPORT_ADMIN] },
    { name: t.nav.accounting, href: '/accounting', icon: FileText, roles: [UserRole.ACCOUNTING_ADMIN, UserRole.SALES_ADMIN] },
    { name: t.nav.marketing, href: '/marketing/banners', icon: Megaphone, roles: [UserRole.CATALOG_ADMIN] },
    { name: 'Team', href: '/team', icon: Users, roles: [UserRole.SUPER_ADMIN, UserRole.ADMIN] },
    { name: 'Plugins', href: '/plugins', icon: Blocks, roles: [UserRole.SUPER_ADMIN, UserRole.ADMIN] },
    { name: t.nav.settings, href: '/settings', icon: Settings, roles: [UserRole.SUPER_ADMIN, UserRole.ADMIN] },
  ];

  // Dynamic Navigation State
  const [dynamicNav, setDynamicNav] = useState<any[]>([]);

  useEffect(() => {
    // Load dynamic navigation from plugins
    // We do a simple fetch here or move to a Context if used widely
    // For simplicity, we fetch once on mount
    const token = localStorage.getItem('admin_token');
    if (token) {
      fetch(`${import.meta.env.VITE_API_URL || 'http://localhost:3001/api'}/admin/plugins/navigation`, {
        headers: { Authorization: `Bearer ${token}` }
      })
        .then(res => res.json())
        .then(data => {
          if (data.success) setDynamicNav(data.data);
        })
        .catch(err => console.error('Failed to load plugin nav', err));
    }
  }, []);

  // Icon Mapper
  const getIcon = (iconName: string) => {
    // Basic mapping (expand as needed)
    const icons: any = { Calculator: FileText, Blocks: Blocks, Settings: Settings };
    return icons[iconName] || Package;
  };

  const combinedNavigation = [
    ...navigation,
    ...dynamicNav.map(item => ({
      name: item.name,
      href: item.href,
      icon: getIcon(item.icon),
      roles: item.roles
    }))
  ];

  const filteredNavigation = combinedNavigation.filter(item => {
    if (!item.roles || item.roles.length === 0) return true;
    return hasRole(item.roles);
  });

  return (
    <div className="min-h-screen bg-gray-50">
      {/* Mobile sidebar backdrop */}
      {sidebarOpen && (
        <div
          className="fixed inset-0 bg-gray-900 bg-opacity-50 z-40 lg:hidden"
          onClick={() => setSidebarOpen(false)}
        />
      )}

      {/* Sidebar */}
      <aside
        className={`fixed top-0 left-0 z-50 h-full w-64 bg-white border-r border-gray-200 transform transition-transform duration-300 ease-in-out lg:translate-x-0 ${sidebarOpen ? 'translate-x-0' : '-translate-x-full'
          }`}
      >
        <div className="flex items-center justify-between h-16 px-6 border-b border-gray-200">
          <div className="flex items-center gap-2">
            <img src={`${import.meta.env.BASE_URL}logo.png`} alt="SMD Logo" className="h-8 w-auto object-contain" />
            <h1 className="text-xl font-bold text-gray-900">SMD</h1>
          </div>
          <button
            onClick={() => setSidebarOpen(false)}
            className="lg:hidden text-gray-500 hover:text-gray-700"
          >
            <X className="w-6 h-6" />
          </button>
        </div>

        <nav className="flex-1 px-4 py-6 space-y-1">
          {filteredNavigation.map((item) => {
            const isActive = location.pathname === item.href;
            return (
              <Link
                key={item.name}
                to={item.href}
                onClick={() => setSidebarOpen(false)}
                className={`flex items-center gap-3 px-4 py-3 rounded-lg transition-colors ${isActive
                  ? 'bg-blue-50 text-blue-600 font-medium'
                  : 'text-gray-700 hover:bg-gray-50'
                  }`}
              >
                <item.icon className="w-5 h-5" />
                {item.name}
              </Link>
            );
          })}
        </nav>

        <div className="border-t border-gray-200 p-4">
          <button
            onClick={onLogout}
            className="flex items-center gap-3 px-4 py-3 w-full text-gray-700 hover:bg-gray-50 rounded-lg transition-colors"
          >
            <LogOut className="w-5 h-5" />
            {t.nav.logout}
          </button>
        </div>
      </aside>

      {/* Main content */}
      <div className="lg:pl-64">
        {/* Top bar */}
        <header className="sticky top-0 z-30 bg-white border-b border-gray-200 h-16 flex items-center px-4 lg:px-8 gap-4">
          <button
            onClick={() => setSidebarOpen(true)}
            className="lg:hidden text-gray-500 hover:text-gray-700 mr-4"
          >
            <Menu className="w-6 h-6" />
          </button>
          <div className="flex-1">
            <h2 className="text-lg font-semibold text-gray-900">
              {navigation.find((item) => item.href === location.pathname)?.name || 'Dashboard'}
            </h2>
          </div>
          <div className="hidden md:flex items-center max-w-md w-full">
            <CatalogSelector />
          </div>
          <button
            onClick={onLogout}
            className="ml-4 flex items-center gap-2 px-4 py-2 text-sm font-medium text-white bg-blue-600 rounded-lg hover:bg-blue-700"
          >
            <LogOut className="w-4 h-4" />
          </button>
        </header>

        {/* Page content */}
        <main className="p-4 lg:p-8">{children}</main>
      </div>
    </div>
  );
}
