import { createContext, useContext, useState, useEffect, ReactNode } from 'react';
import { buildAuthHeaders } from '../utils/apiClient';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

export enum UserRole {
    SUPER_ADMIN = 'SUPER_ADMIN',
    ADMIN = 'ADMIN',
    SALES_ADMIN = 'SALES_ADMIN',
    CATALOG_ADMIN = 'CATALOG_ADMIN',
    ACCOUNTING_ADMIN = 'ACCOUNTING_ADMIN',
    SUPPORT_ADMIN = 'SUPPORT_ADMIN',
    MANAGER = 'MANAGER'
}

interface User {
    id: string;
    email: string;
    firstName: string;
    lastName: string;
    role: UserRole;
    isActive: boolean;
    language?: string;
}

interface AuthContextType {
    user: User | null;
    isAuthenticated: boolean;
    isLoading: boolean;
    login: (token: string, userData: User) => void;
    logout: () => void;
    hasRole: (roles: UserRole[]) => boolean;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

export function AuthProvider({ children }: { children: ReactNode }) {
    const [user, setUser] = useState<User | null>(null);
    const [isLoading, setIsLoading] = useState(true);

    useEffect(() => {
        checkAuth();
    }, []);

    const checkAuth = async () => {
        const token = localStorage.getItem('admin_token');
        if (!token) {
            setIsLoading(false);
            return;
        }

        try {
            const response = await fetch(`${API_URL}/auth/me`, {
                headers: buildAuthHeaders()
            });

            if (response.ok) {
                const data = await response.json();
                if (data.success) {
                    setUser(data.data);
                } else {
                    localStorage.removeItem('admin_token');
                }
            } else {
                localStorage.removeItem('admin_token');
            }
        } catch (error) {
            console.error('Auth verification failed', error);
            localStorage.removeItem('admin_token');
        } finally {
            setIsLoading(false);
        }
    };

    const login = (token: string, userData: User) => {
        localStorage.setItem('admin_token', token);
        setUser(userData);
    };

    const logout = () => {
        localStorage.removeItem('admin_token');
        setUser(null);
        window.location.href = '/login';
    };

    const hasRole = (roles: UserRole[]) => {
        if (!user) return false;
        // SUPER_ADMIN and ADMIN have access to everything
        if (user.role === UserRole.SUPER_ADMIN || user.role === UserRole.ADMIN) return true;
        return roles.includes(user.role);
    };

    return (
        <AuthContext.Provider value={{ user, isAuthenticated: !!user, isLoading, login, logout, hasRole }}>
            {children}
        </AuthContext.Provider>
    );
}

export function useAuth() {
    const context = useContext(AuthContext);
    if (context === undefined) {
        throw new Error('useAuth must be used within an AuthProvider');
    }
    return context;
}
