import { useMemo, useState } from 'react';
import { Plus, Edit2, RefreshCcw, CheckCircle2, AlertTriangle, Loader2 } from 'lucide-react';
import { useCatalog } from '../contexts/CatalogContext';
import { catalogService, CatalogPayload, CatalogSummary } from '../services/catalogService';

interface CatalogFormState {
  name: string;
  slug: string;
  description: string;
  logoUrl: string;
  primaryColor: string;
  isActive: boolean;
  isDefault: boolean;
}

const getInitialFormState = (catalog?: CatalogSummary): CatalogFormState => ({
  name: catalog?.name ?? '',
  slug: catalog?.slug ?? '',
  description: catalog?.description ?? '',
  logoUrl: catalog?.logoUrl ?? '',
  primaryColor: catalog?.primaryColor ?? '#0052cc',
  isActive: catalog?.isActive ?? true,
  isDefault: catalog?.isDefault ?? false,
});

const CatalogModal = ({
  catalog,
  onClose,
  onSubmit,
  saving,
}: {
  catalog: CatalogSummary | null;
  onClose: () => void;
  onSubmit: (payload: CatalogPayload) => Promise<void>;
  saving: boolean;
}) => {
  const [formState, setFormState] = useState<CatalogFormState>(getInitialFormState(catalog ?? undefined));
  const [error, setError] = useState<string | null>(null);

  const handleChange = (
    event: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>
  ) => {
    const { name } = event.target;

    if (event.target instanceof HTMLInputElement && event.target.type === 'checkbox') {
      const { checked } = event.target;
      setFormState(prev => ({
        ...prev,
        [name]: checked,
      }));
      return;
    }

    const { value } = event.target;
    setFormState(prev => ({
      ...prev,
      [name]: value,
    }));
  };

  const handleSubmit = async (event: React.FormEvent) => {
    event.preventDefault();

    if (!formState.name.trim()) {
      setError('Name is required');
      return;
    }

    if (!formState.slug.trim()) {
      setError('Slug is required');
      return;
    }

    try {
      setError(null);
      await onSubmit({
        name: formState.name.trim(),
        slug: formState.slug.trim(),
        description: formState.description.trim() || null,
        logoUrl: formState.logoUrl.trim() || null,
        primaryColor: formState.primaryColor || null,
        isActive: formState.isActive,
        isDefault: formState.isDefault,
      });
    } catch (err) {
      setError(err instanceof Error ? err.message : 'Failed to save catalog');
    }
  };

  return (
    <div className="fixed inset-0 bg-gray-900 bg-opacity-40 flex items-center justify-center z-50">
      <div className="bg-white rounded-2xl shadow-xl w-full max-w-2xl">
        <div className="px-6 py-4 border-b border-gray-100 flex items-center justify-between">
          <div>
            <h2 className="text-xl font-semibold text-gray-900">
              {catalog ? 'Edit Catalog' : 'Create Catalog'}
            </h2>
            <p className="text-sm text-gray-500">Configure catalog visibility and branding</p>
          </div>
          <button onClick={onClose} className="text-gray-400 hover:text-gray-600">✕</button>
        </div>

        <form onSubmit={handleSubmit} className="p-6 space-y-6">
          {error && (
            <div className="flex items-center gap-2 text-sm text-red-600 bg-red-50 border border-red-100 px-3 py-2 rounded-lg">
              <AlertTriangle className="w-4 h-4" />
              {error}
            </div>
          )}

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="text-sm font-medium text-gray-700">Catalog Name</label>
              <input
                type="text"
                name="name"
                value={formState.name}
                onChange={handleChange}
                className="mt-1 w-full rounded-lg border border-gray-300 px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="e.g. Hardware Catalog"
              />
            </div>
            <div>
              <label className="text-sm font-medium text-gray-700">Slug</label>
              <input
                type="text"
                name="slug"
                value={formState.slug}
                onChange={handleChange}
                className="mt-1 w-full rounded-lg border border-gray-300 px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="hardware"
              />
            </div>
          </div>

          <div>
            <label className="text-sm font-medium text-gray-700">Description</label>
            <textarea
              name="description"
              value={formState.description}
              onChange={handleChange}
              rows={3}
              className="mt-1 w-full rounded-lg border border-gray-300 px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
              placeholder="Short summary of this catalog"
            />
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <label className="text-sm font-medium text-gray-700">Logo URL</label>
              <input
                type="url"
                name="logoUrl"
                value={formState.logoUrl}
                onChange={handleChange}
                className="mt-1 w-full rounded-lg border border-gray-300 px-3 py-2 focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                placeholder="https://example.com/logo.png"
              />
            </div>
            <div>
              <label className="text-sm font-medium text-gray-700">Primary Color</label>
              <input
                type="color"
                name="primaryColor"
                value={formState.primaryColor}
                onChange={handleChange}
                className="mt-1 w-16 h-11 rounded border border-gray-200"
              />
            </div>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <label className="flex items-center gap-3 border border-gray-200 rounded-xl px-4 py-3">
              <input
                type="checkbox"
                name="isActive"
                checked={formState.isActive}
                onChange={handleChange}
                className="w-4 h-4 text-blue-600 rounded border-gray-300"
              />
              <div>
                <p className="text-sm font-medium text-gray-900">Catalog is active</p>
                <p className="text-xs text-gray-500">Inactive catalogs stay hidden for customers</p>
              </div>
            </label>

            <label className="flex items-center gap-3 border border-gray-200 rounded-xl px-4 py-3">
              <input
                type="checkbox"
                name="isDefault"
                checked={formState.isDefault}
                onChange={handleChange}
                className="w-4 h-4 text-blue-600 rounded border-gray-300"
              />
              <div>
                <p className="text-sm font-medium text-gray-900">Set as default</p>
                <p className="text-xs text-gray-500">New customers and unassigned requests use this catalog</p>
              </div>
            </label>
          </div>

          <div className="flex items-center justify-end gap-3">
            <button
              type="button"
              onClick={onClose}
              className="px-4 py-2 rounded-lg border border-gray-300 text-gray-700 hover:bg-gray-50"
              disabled={saving}
            >
              Cancel
            </button>
            <button
              type="submit"
              className="px-4 py-2 rounded-lg bg-blue-600 text-white hover:bg-blue-700 flex items-center gap-2"
              disabled={saving}
            >
              {saving && <Loader2 className="w-4 h-4 animate-spin" />}
              {catalog ? 'Save Changes' : 'Create Catalog'}
            </button>
          </div>
        </form>
      </div>
    </div>
  );
};

export default function CatalogsPage() {
  const { catalogs, refreshCatalogs, isLoading, error } = useCatalog();
  const [modalOpen, setModalOpen] = useState(false);
  const [editingCatalog, setEditingCatalog] = useState<CatalogSummary | null>(null);
  const [saving, setSaving] = useState(false);
  const [statusMessage, setStatusMessage] = useState<string | null>(null);

  const stats = useMemo(() => {
    const totalProducts = catalogs.reduce((sum, c) => sum + (c.stats?.products ?? 0), 0);
    const activeCatalogs = catalogs.filter(c => c.isActive).length;
    return { total: catalogs.length, totalProducts, activeCatalogs };
  }, [catalogs]);

  const openCreateModal = () => {
    setEditingCatalog(null);
    setModalOpen(true);
  };

  const openEditModal = (catalog: CatalogSummary) => {
    setEditingCatalog(catalog);
    setModalOpen(true);
  };

  const handleSubmit = async (payload: CatalogPayload) => {
    try {
      setSaving(true);
      if (editingCatalog) {
        await catalogService.update(editingCatalog.id, payload);
        setStatusMessage('Catalog updated');
      } else {
        await catalogService.create(payload);
        setStatusMessage('Catalog created');
      }

      await refreshCatalogs();
      setModalOpen(false);
    } finally {
      setSaving(false);
      setTimeout(() => setStatusMessage(null), 2500);
    }
  };

  const refresh = async () => {
    setStatusMessage('Refreshing...');
    await refreshCatalogs();
    setTimeout(() => setStatusMessage(null), 1500);
  };

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Catalogs</h1>
          <p className="text-sm text-gray-600">Manage the catalogs your customers can access</p>
        </div>
        <div className="flex items-center gap-3">
          <button
            onClick={refresh}
            className="px-4 py-2 border border-gray-200 rounded-lg text-gray-600 hover:bg-gray-50 flex items-center gap-2"
          >
            <RefreshCcw className="w-4 h-4" /> Refresh
          </button>
          <button
            onClick={openCreateModal}
            className="px-4 py-2 bg-blue-600 text-white rounded-lg flex items-center gap-2 hover:bg-blue-700"
          >
            <Plus className="w-4 h-4" />
            New Catalog
          </button>
        </div>
      </div>

      {statusMessage && (
        <div className="rounded-xl border border-blue-100 bg-blue-50 px-4 py-3 text-sm text-blue-700 flex items-center gap-2">
          <CheckCircle2 className="w-4 h-4" />
          {statusMessage}
        </div>
      )}

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <div className="bg-white border border-gray-200 rounded-xl p-4">
          <p className="text-sm text-gray-500">Total Catalogs</p>
          <p className="text-3xl font-semibold text-gray-900">{stats.total}</p>
        </div>
        <div className="bg-white border border-gray-200 rounded-xl p-4">
          <p className="text-sm text-gray-500">Active Catalogs</p>
          <p className="text-3xl font-semibold text-gray-900">{stats.activeCatalogs}</p>
        </div>
        <div className="bg-white border border-gray-200 rounded-xl p-4">
          <p className="text-sm text-gray-500">Total Products</p>
          <p className="text-3xl font-semibold text-gray-900">{stats.totalProducts}</p>
        </div>
      </div>

      <div className="bg-white rounded-xl border border-gray-200 overflow-hidden">
        {isLoading ? (
          <div className="flex items-center justify-center py-16 text-gray-500">
            <Loader2 className="w-5 h-5 animate-spin mr-2" /> Loading catalogs...
          </div>
        ) : error ? (
          <div className="p-6 text-red-600">{error}</div>
        ) : catalogs.length === 0 ? (
          <div className="p-12 text-center">
            <p className="text-lg font-semibold text-gray-900">No catalogs yet</p>
            <p className="text-sm text-gray-500 mt-2">Create your first catalog to get started.</p>
            <button
              onClick={openCreateModal}
              className="mt-4 px-4 py-2 bg-blue-600 text-white rounded-lg"
            >
              Create Catalog
            </button>
          </div>
        ) : (
          <div className="overflow-x-auto">
            <table className="min-w-full divide-y divide-gray-200">
              <thead className="bg-gray-50">
                <tr>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Catalog</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Stats</th>
                  <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                  <th className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                </tr>
              </thead>
              <tbody className="bg-white divide-y divide-gray-200">
                {catalogs.map(catalog => (
                  <tr key={catalog.id} className="hover:bg-gray-50">
                    <td className="px-6 py-4">
                      <div className="flex flex-col">
                        <span className="font-semibold text-gray-900">{catalog.name}</span>
                        <span className="text-sm text-gray-500">/{catalog.slug}</span>
                        {catalog.description && (
                          <span className="text-xs text-gray-500 mt-1">{catalog.description}</span>
                        )}
                      </div>
                    </td>
                    <td className="px-6 py-4 text-sm text-gray-600">
                      <div className="flex flex-wrap gap-2">
                        <span className="inline-flex items-center px-2 py-0.5 rounded-full bg-blue-50 text-blue-700 text-xs">
                          {(catalog.stats?.products ?? 0)} products
                        </span>
                        <span className="inline-flex items-center px-2 py-0.5 rounded-full bg-purple-50 text-purple-700 text-xs">
                          {(catalog.stats?.categories ?? 0)} categories
                        </span>
                        <span className="inline-flex items-center px-2 py-0.5 rounded-full bg-orange-50 text-orange-700 text-xs">
                          {(catalog.stats?.customers ?? 0)} customers
                        </span>
                      </div>
                    </td>
                    <td className="px-6 py-4">
                      <div className="flex flex-col gap-1 text-sm">
                        <span
                          className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                            catalog.isActive
                              ? 'bg-green-100 text-green-800'
                              : 'bg-gray-200 text-gray-700'
                          }`}
                        >
                          {catalog.isActive ? 'Active' : 'Inactive'}
                        </span>
                        {catalog.isDefault && (
                          <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">
                            Default
                          </span>
                        )}
                      </div>
                    </td>
                    <td className="px-6 py-4 text-right">
                      <button
                        onClick={() => openEditModal(catalog)}
                        className="inline-flex items-center gap-2 px-3 py-2 text-sm text-blue-600 hover:bg-blue-50 rounded-lg"
                      >
                        <Edit2 className="w-4 h-4" /> Edit
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>
        )}
      </div>

      {modalOpen && (
        <CatalogModal
          catalog={editingCatalog}
          onClose={() => setModalOpen(false)}
          onSubmit={handleSubmit}
          saving={saving}
        />
      )}
    </div>
  );
}
