import { PrismaClient } from '@prisma/client';
import { resolveCatalogContext } from '../utils/catalogContext.js';
const prisma = new PrismaClient();
const buildOrderCatalogFilter = (catalogId) => ({
    items: {
        some: {
            product: {
                catalogId,
            },
        },
    },
});
const calculateChange = (current, previous) => {
    if (previous === 0) {
        return current > 0 ? 100 : 0;
    }
    return ((current - previous) / previous) * 100;
};
export const getDashboardStats = async (req, res, next) => {
    try {
        const catalog = await resolveCatalogContext(req, prisma);
        const now = new Date();
        const thirtyDaysAgo = new Date(now);
        thirtyDaysAgo.setDate(now.getDate() - 30);
        const sixtyDaysAgo = new Date(now);
        sixtyDaysAgo.setDate(now.getDate() - 60);
        const orderFilter = buildOrderCatalogFilter(catalog.id);
        const [totalRevenueAgg, totalOrders, totalProducts, totalCustomers, revenueCurrentAgg, revenuePreviousAgg, ordersCurrent, ordersPrevious, productsCurrent, productsPrevious, customersCurrent, customersPrevious,] = await Promise.all([
            prisma.order.aggregate({
                _sum: { totalAmount: true },
                where: orderFilter,
            }),
            prisma.order.count({ where: orderFilter }),
            prisma.product.count({ where: { catalogId: catalog.id } }),
            prisma.catalogCustomer.count({ where: { catalogId: catalog.id } }),
            prisma.order.aggregate({
                _sum: { totalAmount: true },
                where: {
                    ...orderFilter,
                    createdAt: { gte: thirtyDaysAgo, lt: now },
                },
            }),
            prisma.order.aggregate({
                _sum: { totalAmount: true },
                where: {
                    ...orderFilter,
                    createdAt: { gte: sixtyDaysAgo, lt: thirtyDaysAgo },
                },
            }),
            prisma.order.count({
                where: {
                    ...orderFilter,
                    createdAt: { gte: thirtyDaysAgo, lt: now },
                },
            }),
            prisma.order.count({
                where: {
                    ...orderFilter,
                    createdAt: { gte: sixtyDaysAgo, lt: thirtyDaysAgo },
                },
            }),
            prisma.product.count({
                where: {
                    catalogId: catalog.id,
                    createdAt: { gte: thirtyDaysAgo, lt: now },
                },
            }),
            prisma.product.count({
                where: {
                    catalogId: catalog.id,
                    createdAt: { gte: sixtyDaysAgo, lt: thirtyDaysAgo },
                },
            }),
            prisma.catalogCustomer.count({
                where: {
                    catalogId: catalog.id,
                    assignedAt: { gte: thirtyDaysAgo, lt: now },
                },
            }),
            prisma.catalogCustomer.count({
                where: {
                    catalogId: catalog.id,
                    assignedAt: { gte: sixtyDaysAgo, lt: thirtyDaysAgo },
                },
            }),
        ]);
        const totalRevenue = Number(totalRevenueAgg._sum.totalAmount ?? 0);
        const revenueCurrent = Number(revenueCurrentAgg._sum.totalAmount ?? 0);
        const revenuePrevious = Number(revenuePreviousAgg._sum.totalAmount ?? 0);
        res.json({
            success: true,
            data: {
                totalRevenue,
                revenueChange: calculateChange(revenueCurrent, revenuePrevious),
                totalOrders,
                ordersChange: calculateChange(ordersCurrent, ordersPrevious),
                totalProducts,
                productsChange: calculateChange(productsCurrent, productsPrevious),
                totalCustomers,
                customersChange: calculateChange(customersCurrent, customersPrevious),
            },
        });
    }
    catch (error) {
        next(error);
    }
};
export const getRecentOrders = async (req, res, next) => {
    try {
        const catalog = await resolveCatalogContext(req, prisma);
        const limit = Number(req.query.limit) || 5;
        const orders = await prisma.order.findMany({
            where: buildOrderCatalogFilter(catalog.id),
            orderBy: { createdAt: 'desc' },
            take: limit,
            select: {
                id: true,
                orderNumber: true,
                customerName: true,
                totalAmount: true,
                status: true,
                createdAt: true,
            },
        });
        res.json({
            success: true,
            data: orders,
        });
    }
    catch (error) {
        next(error);
    }
};
export const getLowStockProducts = async (req, res, next) => {
    try {
        const catalog = await resolveCatalogContext(req, prisma);
        const limit = Number(req.query.limit) || 5;
        const threshold = Number(req.query.threshold) || 5;
        const products = await prisma.product.findMany({
            where: {
                catalogId: catalog.id,
                stockQuantity: { lte: threshold },
                isActive: true,
            },
            orderBy: { stockQuantity: 'asc' },
            take: limit,
            select: {
                id: true,
                name: true,
                sku: true,
                stockQuantity: true,
            },
        });
        const formatted = products.map(product => ({
            id: product.id,
            name: product.name,
            sku: product.sku ?? '',
            stock: product.stockQuantity,
            minStock: threshold,
        }));
        res.json({
            success: true,
            data: formatted,
        });
    }
    catch (error) {
        next(error);
    }
};
//# sourceMappingURL=dashboard.controller.js.map