import { PrismaClient } from '@prisma/client';
const prisma = new PrismaClient();
/**
 * Helper function to map order status to invoice status
 */
function mapOrderStatusToInvoiceStatus(orderStatus, paymentStatus) {
    if (paymentStatus === 'PAID')
        return 'PAID';
    if (orderStatus === 'CANCELLED')
        return 'CANCELLED';
    return 'ISSUED';
}
/**
 * Get all invoices for the logged-in customer
 * Maps orders to invoice format
 */
export const getMyInvoices = async (req, res, next) => {
    try {
        const userId = req.user?.id;
        if (!userId) {
            return res.status(401).json({
                success: false,
                error: { message: 'Unauthorized' },
            });
        }
        const orders = await prisma.order.findMany({
            where: { userId },
            include: {
                items: true,
                user: {
                    select: {
                        id: true,
                        email: true,
                        firstName: true,
                        lastName: true,
                        companyName: true,
                    },
                },
            },
            orderBy: { createdAt: 'desc' },
        });
        // Map orders to invoice format with items
        const invoices = orders.map((order) => ({
            id: order.id,
            invoiceNumber: order.orderNumber,
            orderId: order.id,
            userId: order.userId,
            quotationId: order.quotationId || undefined,
            // TVA fields
            totalBrutHT: order.totalBrutHT,
            totalDiscount: order.totalDiscount,
            totalNetHT: order.totalNetHT,
            tva19Amount: order.tva19Amount,
            tva7Amount: order.tva7Amount,
            tva0Amount: order.tva0Amount,
            totalTVA: order.totalTVA,
            netAPayer: order.netAPayer,
            // Deprecated fields (for backward compatibility)
            subtotal: order.subtotal,
            taxAmount: order.taxAmount,
            totalAmount: order.totalAmount,
            status: mapOrderStatusToInvoiceStatus(order.status, order.paymentStatus),
            paymentTerm: order.paymentTerm || undefined,
            dueDate: order.dueDate?.toISOString() || undefined,
            paidDate: order.paymentStatus === 'PAID' ? order.updatedAt.toISOString() : undefined,
            paidAmount: order.paymentStatus === 'PAID' ? Number(order.totalAmount) : undefined,
            issuedDate: order.createdAt.toISOString(),
            createdAt: order.createdAt.toISOString(),
            updatedAt: order.updatedAt.toISOString(),
            notes: order.notes || undefined,
            user: order.user,
            // Include order items
            items: order.items.map(item => ({
                id: item.id,
                productName: item.productName,
                productSku: item.productSku || '',
                quantity: item.quantity,
                unitPrice: Number(item.unitPrice),
                discount: Number(item.discount || 0),
                totalHT: Number(item.totalHT),
                tvaRate: Number(item.tvaRate),
                tvaAmount: Number(item.tvaAmount),
                totalTTC: Number(item.totalTTC),
                tvaExempt: item.tvaExempt,
            })),
        }));
        res.json({
            success: true,
            data: invoices,
        });
    }
    catch (error) {
        next(error);
    }
};
/**
 * Get invoice by ID
 */
export const getInvoiceById = async (req, res, next) => {
    try {
        const { id } = req.params;
        const userId = req.user?.id;
        const order = await prisma.order.findUnique({
            where: { id },
            include: {
                items: true,
                user: {
                    select: {
                        id: true,
                        email: true,
                        firstName: true,
                        lastName: true,
                        companyName: true,
                    },
                },
            },
        });
        if (!order) {
            return res.status(404).json({
                success: false,
                error: { message: 'Invoice not found' },
            });
        }
        // Check ownership
        if (order.userId !== userId && req.user?.role === 'CUSTOMER') {
            return res.status(403).json({
                success: false,
                error: { message: 'Access denied' },
            });
        }
        // Map Order to Invoice
        const invoice = {
            id: order.id,
            invoiceNumber: order.orderNumber,
            orderId: order.id,
            userId: order.userId,
            quotationId: order.quotationId || undefined,
            // TVA fields
            totalBrutHT: order.totalBrutHT,
            totalDiscount: order.totalDiscount,
            totalNetHT: order.totalNetHT,
            tva19Amount: order.tva19Amount,
            tva7Amount: order.tva7Amount,
            tva0Amount: order.tva0Amount,
            totalTVA: order.totalTVA,
            netAPayer: order.netAPayer,
            // Deprecated fields (for backward compatibility)
            subtotal: order.subtotal,
            taxAmount: order.taxAmount,
            totalAmount: order.totalAmount,
            status: mapOrderStatusToInvoiceStatus(order.status, order.paymentStatus),
            paymentTerm: order.paymentTerm || undefined,
            dueDate: order.dueDate?.toISOString() || undefined,
            paidDate: order.paymentStatus === 'PAID' ? order.updatedAt.toISOString() : undefined,
            paidAmount: order.paymentStatus === 'PAID' ? Number(order.totalAmount) : undefined,
            issuedDate: order.createdAt.toISOString(),
            createdAt: order.createdAt.toISOString(),
            updatedAt: order.updatedAt.toISOString(),
            notes: order.notes || undefined,
            user: order.user,
            // Include order items
            items: order.items.map(item => ({
                id: item.id,
                productName: item.productName,
                productSku: item.productSku || '',
                quantity: item.quantity,
                unitPrice: Number(item.unitPrice),
                discount: Number(item.discount || 0),
                totalHT: Number(item.totalHT),
                tvaRate: Number(item.tvaRate),
                tvaAmount: Number(item.tvaAmount),
                totalTTC: Number(item.totalTTC),
                tvaExempt: item.tvaExempt,
            })),
        };
        res.json({
            success: true,
            data: invoice,
        });
    }
    catch (error) {
        next(error);
    }
};
//# sourceMappingURL=invoice.controller.js.map