/**
 * Core Hook System (Event Bus)
 *
 * Implements a "Waterfull" or "Pipeline" pattern where plugins can intercept,
 * modify, or augment data flow.
 */
export class HookSystem {
    hooks = new Map();
    /**
     * Register a handler for a specific hook
     * @param hookName The name of the containment point (e.g. 'order.validate')
     * @param handler The function to execute
     */
    on(hookName, handler) {
        if (!this.hooks.has(hookName)) {
            this.hooks.set(hookName, []);
        }
        this.hooks.get(hookName)?.push(handler);
        console.log(`[HookSystem] Registered handler for '${hookName}'`);
    }
    /**
     * Trigger a hook. Handlers are executed sequentially.
     * @param hookName
     * @param context The mutable context object passed through handlers
     * @returns The modified context
     */
    async trigger(hookName, context) {
        const handlers = this.hooks.get(hookName) || [];
        if (handlers.length === 0) {
            return context;
        }
        // Execute handlers in sequence (Middleware pattern)
        // We compose them so that each handler calls 'next' to proceed
        const dispatch = async (index) => {
            if (index >= handlers.length)
                return;
            const handler = handlers[index];
            await handler(context, async () => {
                await dispatch(index + 1);
            });
        };
        await dispatch(0);
        return context;
    }
}
export const hooks = new HookSystem();
//# sourceMappingURL=HookSystem.js.map