import { PrismaClient } from '@prisma/client';
import bcrypt from 'bcryptjs';
const prisma = new PrismaClient();
async function seedUsers() {
    console.log('Seeding Test Users...');
    try {
        // 1. Create Catalog
        const catalog = await prisma.catalog.upsert({
            where: { id: 'CAT_DEFAULT' },
            create: {
                id: 'CAT_DEFAULT',
                name: 'General Catalog',
                slug: 'general',
                isActive: true,
                isDefault: true
            },
            update: {}
        });
        console.log('Catalog ensured.');
        // 2. Create Admin User
        const adminPassword = await bcrypt.hash('admin123', 10);
        const admin = await prisma.user.upsert({
            where: { email: 'admin@smd.com' },
            create: {
                email: 'admin@smd.com',
                passwordHash: adminPassword,
                firstName: 'Admin',
                lastName: 'User',
                role: 'ADMIN',
                isActive: true
            },
            update: {
                passwordHash: adminPassword,
                role: 'ADMIN',
                isActive: true
            },
        });
        console.log('Admin user seeded: admin@smd.com / admin123');
        // 3. Create Customer User
        const customerPassword = await bcrypt.hash('customer123', 10);
        const customer = await prisma.user.upsert({
            where: { email: 'customer@smd.com' },
            create: {
                email: 'customer@smd.com',
                passwordHash: customerPassword,
                firstName: 'John',
                lastName: 'Customer',
                role: 'CUSTOMER',
                isActive: true,
                catalogAssignments: {
                    create: {
                        catalogId: catalog.id
                    }
                }
            },
            update: {
                passwordHash: customerPassword,
                role: 'CUSTOMER',
                isActive: true
            },
        });
        // Ensure catalog assignment exists if user existed
        const assignment = await prisma.catalogCustomer.findFirst({
            where: { customerId: customer.id, catalogId: catalog.id }
        });
        if (!assignment) {
            await prisma.catalogCustomer.create({
                data: {
                    customerId: customer.id,
                    catalogId: catalog.id
                }
            });
        }
        console.log('Customer user seeded: customer@smd.com / customer123');
        // 4. Create/Reset a Test Product
        const product = await prisma.product.upsert({
            where: { sku: 'BROWSER-TEST-1' },
            create: {
                sku: 'BROWSER-TEST-1',
                name: 'Browser Test Widget',
                slug: 'browser-test-widget',
                basePrice: 50.0,
                stockQuantity: 10,
                catalogId: catalog.id,
                categoryId: null // Assuming optional or we skip category for now
                // Note: Schema might require categoryId if not optional in DB but optional in type. 
                // Checking schema... categoryId is String (required) in Product model line 233.
                // We need a category.
            },
            update: {
                stockQuantity: 10 // Reset stock
            }
        });
        // Wait, the previous upsert might fail if categoryId is missing in create.
        // I should create category first.
    }
    catch (error) {
        if (error.code === 'P2002' || error.message?.includes('categoryId')) {
            console.log('Product upsert failed, likely due to missing Category. Retrying with Category...');
            // Retry logic in catch block is messy, let's just make sure we do it right.
        }
        else {
            console.error('Seeding Failed:', error);
        }
    }
    finally {
        await prisma.$disconnect();
    }
}
// Better version of main
async function main() {
    const prisma = new PrismaClient();
    try {
        console.log('Starting seed...');
        // Catalog
        const catalog = await prisma.catalog.upsert({
            where: { id: 'CAT_BROWSER' },
            create: { id: 'CAT_BROWSER', name: 'Browser Catalog', slug: 'browser', isActive: true, isDefault: true, defaultTvaRate: 19 },
            update: {}
        });
        // Category
        const category = await prisma.category.upsert({
            where: { catalogId_slug: { catalogId: catalog.id, slug: 'browser-cat' } },
            create: { catalogId: catalog.id, name: 'Browser Category', slug: 'browser-cat', isActive: true },
            update: {}
        });
        // Product
        const product = await prisma.product.upsert({
            where: { catalogId_sku: { catalogId: catalog.id, sku: 'BROWSER-SKU-1' } },
            create: {
                sku: 'BROWSER-SKU-1',
                name: 'Test Widget',
                slug: 'test-widget',
                basePrice: 100,
                stockQuantity: 10,
                catalogId: catalog.id,
                categoryId: category.id,
                isActive: true
            },
            update: { stockQuantity: 10 }
        });
        // Passwords
        const adminHash = await bcrypt.hash('admin123', 10);
        const customerHash = await bcrypt.hash('customer123', 10);
        // Updates
        await prisma.user.upsert({
            where: { email: 'admin@smd.com' },
            create: { email: 'admin@smd.com', passwordHash: adminHash, role: 'ADMIN', firstName: 'Admin', lastName: 'Test' },
            update: { passwordHash: adminHash, role: 'ADMIN' }
        });
        const customer = await prisma.user.upsert({
            where: { email: 'customer@smd.com' },
            create: { email: 'customer@smd.com', passwordHash: customerHash, role: 'CUSTOMER', firstName: 'Customer', lastName: 'Test' },
            update: { passwordHash: customerHash, role: 'CUSTOMER' }
        });
        // Assignment
        const assign = await prisma.catalogCustomer.findUnique({
            where: { catalogId_customerId: { catalogId: catalog.id, customerId: customer.id } }
        });
        if (!assign) {
            await prisma.catalogCustomer.create({ data: { catalogId: catalog.id, customerId: customer.id } });
        }
        console.log('Seeds successful.');
        console.log('Customer: customer@smd.com / customer123');
        console.log('Admin: admin@smd.com / admin123');
        console.log('Product: Test Widget (10 in stock)');
    }
    catch (e) {
        console.error(e);
    }
    finally {
        await prisma.$disconnect();
    }
}
main();
//# sourceMappingURL=seed-test-users.js.map