/**
 * Discount calculation utilities for product pricing
 */
/**
 * Calculate product discount and final price
 *
 * @param basePrice - Original product price
 * @param hasDiscount - Whether discount is enabled
 * @param discountType - Type of discount: "percentage" or "fixed"
 * @param discountValue - Discount value (percentage or fixed amount)
 * @param discountStartDate - Optional start date for discount
 * @param discountEndDate - Optional end date for discount
 * @returns DiscountInfo object with all calculated values
 */
export function calculateProductDiscount(basePrice, hasDiscount, discountType, discountValue, discountStartDate, discountEndDate) {
    const price = typeof basePrice === 'number' ? basePrice : Number(basePrice);
    const now = new Date();
    // Check if discount is active
    const isDateValid = (!discountStartDate || new Date(discountStartDate) <= now) &&
        (!discountEndDate || new Date(discountEndDate) >= now);
    const isActive = hasDiscount &&
        discountValue != null &&
        Number(discountValue) > 0 &&
        isDateValid;
    if (!isActive) {
        return {
            hasDiscount: false,
            originalPrice: price,
            discountAmount: 0,
            finalPrice: price,
            savings: 0,
            isActive: false,
        };
    }
    const value = typeof discountValue === 'number' ? discountValue : Number(discountValue);
    let discountAmount = 0;
    if (discountType === 'percentage') {
        // Percentage discount: value is percentage (e.g., 15 for 15%)
        discountAmount = (price * value) / 100;
    }
    else if (discountType === 'fixed') {
        // Fixed amount discount: cap at base price to avoid negative
        discountAmount = Math.min(value, price);
    }
    const finalPrice = Math.max(0, price - discountAmount);
    return {
        hasDiscount: true,
        discountType: discountType,
        discountValue: value,
        originalPrice: price,
        discountAmount,
        finalPrice,
        savings: discountAmount,
        isActive: true,
    };
}
/**
 * Format discount for display (e.g., "15.0 %" or "50.00 DT")
 */
export function formatDiscount(discountType, discountValue) {
    if (!discountValue)
        return '0.00';
    const value = typeof discountValue === 'number' ? discountValue : Number(discountValue);
    if (discountType === 'percentage') {
        return `${value.toFixed(1)} %`;
    }
    else {
        return `${value.toFixed(2)} DT`;
    }
}
/**
 * Calculate discount amount for a line item
 * Used in quotations and orders
 */
export function calculateLineDiscount(quantity, unitPrice, discountType, discountValue) {
    if (!discountValue || !discountType)
        return 0;
    const value = typeof discountValue === 'number' ? discountValue : Number(discountValue);
    const totalBrut = quantity * unitPrice;
    if (discountType === 'percentage') {
        return (totalBrut * value) / 100;
    }
    else if (discountType === 'fixed') {
        // Fixed discount applies per unit, then multiply by quantity
        return Math.min(value * quantity, totalBrut);
    }
    return 0;
}
/**
 * Check if a discount is currently active
 */
export function isDiscountActive(hasDiscount, discountStartDate, discountEndDate) {
    if (!hasDiscount)
        return false;
    const now = new Date();
    if (discountStartDate && new Date(discountStartDate) > now) {
        return false;
    }
    if (discountEndDate && new Date(discountEndDate) < now) {
        return false;
    }
    return true;
}
//# sourceMappingURL=discountCalculations.js.map