import { PrismaClient } from '@prisma/client';

const prisma = new PrismaClient();

async function fixCatalogAssignments() {
    console.log('=== FIXING CATALOG ASSIGNMENTS ===\n');

    // Get default catalog
    const defaultCatalog = await prisma.catalog.findFirst({
        where: { isDefault: true }
    });

    if (!defaultCatalog) {
        console.error('❌ No default catalog found! Please create one first.');
        return;
    }

    console.log(`Using default catalog: ${defaultCatalog.name} (${defaultCatalog.id})\n`);

    // Fix categories without catalog
    const categoriesFixed = await prisma.category.updateMany({
        where: { catalogId: null },
        data: { catalogId: defaultCatalog.id }
    });
    console.log(`✓ Fixed ${categoriesFixed.count} categories with NULL catalog_id`);

    // Fix products without catalog
    const productsFixed = await prisma.product.updateMany({
        where: { catalogId: null },
        data: { catalogId: defaultCatalog.id }
    });
    console.log(`✓ Fixed ${productsFixed.count} products with NULL catalog_id`);

    // Assign customers without catalog to default
    const customers = await prisma.user.findMany({
        where: {
            role: 'CUSTOMER',
            catalogAssignments: { none: {} }
        }
    });

    for (const customer of customers) {
        await prisma.catalogCustomer.create({
            data: {
                catalogId: defaultCatalog.id,
                customerId: customer.id
            }
        });
        console.log(`✓ Assigned ${customer.email} to ${defaultCatalog.name}`);
    }

    console.log(`\n✅ Fixed ${customers.length} customers without catalog assignment`);
    console.log('\nRun diagnose-catalog-isolation.ts to verify the fixes.');
}

fixCatalogAssignments()
    .catch(console.error)
    .finally(() => prisma.$disconnect());
