import { Request } from 'express';
import { PrismaClient } from '@prisma/client';
import { createError } from '../middleware/errorHandler.js';

export interface CatalogContext {
  id: string;
  slug: string;
  name: string;
  isDefault: boolean;
}

interface CatalogContextOptions {
  required?: boolean;
  fallbackToDefault?: boolean;
  allowBody?: boolean;
  useCustomerAssignment?: boolean;
  requireCustomerAssignment?: boolean;
}

const getStringValue = (value: unknown): string | undefined => {
  if (typeof value === 'string' && value.trim().length > 0) {
    return value.trim();
  }
  return undefined;
};

export const ensureCustomerAssignedToCatalog = async (
  prisma: PrismaClient,
  customerId: string,
  catalogId: string
) => {
  const assignment = await prisma.catalogCustomer.findFirst({
    where: {
      catalogId,
      customerId,
    },
  });

  if (!assignment) {
    throw createError('Customer does not belong to this catalog', 403);
  }

  return assignment;
};

export const resolveCatalogContext = async (
  req: Request,
  prisma: PrismaClient,
  options: CatalogContextOptions = {}
): Promise<CatalogContext> => {
  const {
    required = true,
    fallbackToDefault = true,
    allowBody = false,
    useCustomerAssignment = true,
    requireCustomerAssignment = false,
  } = options;

  const headerCatalogId = getStringValue(req.header('x-catalog-id'));
  const headerCatalogSlug = getStringValue(req.header('x-catalog-slug'));
  const queryCatalogId = getStringValue(req.query.catalogId as string | undefined);
  const queryCatalogSlug = getStringValue(req.query.catalogSlug as string | undefined);
  const bodyCatalogId = allowBody ? getStringValue((req.body || {}).catalogId) : undefined;
  const bodyCatalogSlug = allowBody ? getStringValue((req.body || {}).catalogSlug) : undefined;

  const requestedCatalogId = headerCatalogId || bodyCatalogId || queryCatalogId;
  const requestedCatalogSlug = headerCatalogSlug || bodyCatalogSlug || queryCatalogSlug;

  let catalog = null;

  if (requestedCatalogId) {
    catalog = await prisma.catalog.findUnique({ where: { id: requestedCatalogId } });
  }

  if (!catalog && requestedCatalogSlug) {
    catalog = await prisma.catalog.findUnique({ where: { slug: requestedCatalogSlug } });
  }

  const user = (req as any).user as { id: string; role: string } | undefined;

  if (!catalog && useCustomerAssignment && user?.role === 'CUSTOMER') {
    const assignment = await prisma.catalogCustomer.findFirst({
      where: { customerId: user.id },
      include: { catalog: true },
      orderBy: { assignedAt: 'desc' },
    });

    if (assignment?.catalog) {
      catalog = assignment.catalog;
    }
  }

  if (!catalog && fallbackToDefault) {
    catalog = await prisma.catalog.findFirst({ where: { isDefault: true } });
  }

  if (!catalog) {
    if (required) {
      throw createError('Unable to resolve catalog context', 400);
    }

    throw createError('Catalog not found', 404);
  }

  if (!catalog.isActive) {
    throw createError('Catalog is inactive', 403);
  }

  if (requireCustomerAssignment && user?.role === 'CUSTOMER') {
    await ensureCustomerAssignedToCatalog(prisma, user.id, catalog.id);
  }

  return {
    id: catalog.id,
    slug: catalog.slug,
    name: catalog.name,
    isDefault: catalog.isDefault,
  };
};
