import React, { createContext, useContext, useState, useEffect, ReactNode } from 'react';

interface User {
  id: string;
  email: string;
  username?: string;
  firstName?: string;
  lastName?: string;
  phone?: string;
  companyName?: string;
  customerType?: string;
  role: string;
  catalogId?: string; // Added catalogId
  // B2B Financial fields
  paymentTerm?: string;
  financialLimit?: number;
  currentOutstanding?: number;
  outstandingBalance?: number; // Keep for backward compatibility
  accountStatus?: string;
  paymentMethod?: string; // Keep for backward compatibility if needed
  paymentTerms?: string; // Keep for backward compatibility if needed
}


interface AuthContextType {
  user: User | null;
  token: string | null;
  isAuthenticated: boolean;
  isLoading: boolean;
  login: (credentials: { email?: string; username?: string; password: string }) => Promise<void>;
  logout: () => void;
  refreshUser: () => Promise<void>;
  catalogId: string | null; // Expose catalogId
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

const API_URL = import.meta.env.VITE_API_URL || 'http://catalogquienquillerie.sqb-tunisie.com:3001/api';

export function AuthProvider({ children }: { children: ReactNode }) {
  const [user, setUser] = useState<User | null>(null);
  const [token, setToken] = useState<string | null>(null);
  const [isLoading, setIsLoading] = useState(true);

  // Load user from localStorage on mount
  useEffect(() => {
    const loadUser = async () => {
      try {
        const storedToken = localStorage.getItem('customer_token');
        const storedUser = localStorage.getItem('customer_user');

        if (storedToken && storedUser) {
          setToken(storedToken);
          setUser(JSON.parse(storedUser));

          // Verify token is still valid
          await refreshUser(storedToken);
        }
      } catch (error) {
        console.error('Failed to load user:', error);
        // Clear invalid data
        localStorage.removeItem('customer_token');
        localStorage.removeItem('customer_user');
        localStorage.removeItem('customer_refresh_token');
      } finally {
        setIsLoading(false);
      }
    };

    loadUser();
  }, []);

  const login = async (credentials: { email?: string; username?: string; password: string }) => {
    try {
      const response = await fetch(`${API_URL}/auth/login`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(credentials),
      });

      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error?.message || 'Login failed');
      }

      const data = await response.json();

      if (data.success && data.data) {
        const { user: userData, token: authToken, refreshToken } = data.data;

        // Only allow CUSTOMER role to login on frontend
        if (userData.role !== 'CUSTOMER') {
          throw new Error('Access denied. Please use the admin dashboard.');
        }

        // Extract catalogId from assignments if available and not directly on user object
        // The backend login response should ideally include catalogId directly or via assignments
        // For now, we assume the backend might send it, or we rely on assignments if present
        // If not present, we might need to fetch it or rely on backend context

        // Ensure catalogId is part of userData if it comes from assignments
        if (!userData.catalogId && userData.catalogAssignments && userData.catalogAssignments.length > 0) {
          userData.catalogId = userData.catalogAssignments[0].catalogId;
        }

        setUser(userData);
        setToken(authToken);

        // Store in localStorage
        localStorage.setItem('customer_token', authToken);
        localStorage.setItem('customer_user', JSON.stringify(userData));
        localStorage.setItem('customer_refresh_token', refreshToken);

      } else {
        throw new Error('Invalid response from server');
      }
    } catch (error) {
      console.error('Login error:', error);
      throw error;
    }
  };

  const logout = async () => {
    try {
      const refreshToken = localStorage.getItem('customer_refresh_token');

      if (token && refreshToken) {
        await fetch(`${API_URL}/auth/logout`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${token}`,
          },
          body: JSON.stringify({ refreshToken }),
        });
      }
    } catch (error) {
      console.error('Logout error:', error);
    } finally {
      // Preserve user preferences before clearing auth data
      const savedLanguage = localStorage.getItem('customerLanguage');

      // Clear state and localStorage
      setUser(null);
      setToken(null);
      localStorage.removeItem('customer_token');
      localStorage.removeItem('customer_user');
      localStorage.removeItem('customer_refresh_token');

      // Restore user preferences
      if (savedLanguage) {
        localStorage.setItem('customerLanguage', savedLanguage);
      }
    }
  };

  const refreshUser = async (authToken?: string) => {
    const tokenToUse = authToken || token;

    if (!tokenToUse) {
      return;
    }

    try {
      const response = await fetch(`${API_URL}/auth/me`, {
        headers: {
          'Authorization': `Bearer ${tokenToUse}`,
        },
      });

      // Explicit auth failure: clear state and storage
      if (response.status === 401 || response.status === 403) {
        setUser(null);
        setToken(null);
        localStorage.removeItem('customer_token');
        localStorage.removeItem('customer_user');
        localStorage.removeItem('customer_refresh_token');
        return;
      }

      // Other HTTP errors: log and keep existing user/token (avoid logging customer out)
      if (!response.ok) {
        console.error('Failed to refresh user:', response.status);
        return;
      }

      const data = await response.json();

      if (data.success && data.data) {
        setUser(data.data);
        localStorage.setItem('customer_user', JSON.stringify(data.data));
      }
    } catch (error) {
      // Network or unexpected errors: log, but do not clear auth
      console.error('Refresh user error:', error);
    }
  };

  const value: AuthContextType = {
    user,
    token,
    isAuthenticated: !!user && !!token,
    isLoading,
    login,
    logout,
    refreshUser,
    catalogId: user?.catalogId || null,
  };

  return <AuthContext.Provider value={value}>{children}</AuthContext.Provider>;
}

export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider');
  }
  return context;
}

