import { buildAuthHeaders } from '../utils/apiClient';

const API_URL = import.meta.env.VITE_API_URL || 'http://localhost:3001/api';

export interface CatalogSummary {
  id: string;
  name: string;
  slug: string;
  description?: string;
  logoUrl?: string | null;
  primaryColor?: string | null;
  isActive: boolean;
  isDefault: boolean;
  stats?: {
    categories?: number;
    products?: number;
    banners?: number;
    customers?: number;
  };
  createdAt?: string;
  updatedAt?: string;
}

export interface CatalogPayload {
  name: string;
  slug: string;
  description?: string | null;
  logoUrl?: string | null;
  primaryColor?: string | null;
  isActive?: boolean;
  isDefault?: boolean;
}

const parseCatalogResponse = async (response: Response): Promise<CatalogSummary[]> => {
  if (!response.ok) {
    const error = await response.json().catch(() => ({}));
    throw new Error(error.error?.message || 'Failed to fetch catalogs');
  }

  const payload = await response.json();

  if (Array.isArray(payload)) {
    return payload;
  }

  if (payload?.success && Array.isArray(payload.data)) {
    return payload.data as CatalogSummary[];
  }

  if (Array.isArray(payload?.data?.data)) {
    return payload.data.data as CatalogSummary[];
  }

  return [];
};

const parseSingleCatalog = async (response: Response): Promise<CatalogSummary> => {
  if (!response.ok) {
    const error = await response.json().catch(() => ({}));
    throw new Error(error.error?.message || 'Catalog request failed');
  }

  const payload = await response.json();

  if (payload?.data) {
    return payload.data as CatalogSummary;
  }

  return payload as CatalogSummary;
};

export const catalogService = {
  async getAll(): Promise<CatalogSummary[]> {
    const response = await fetch(`${API_URL}/catalogs`, {
      headers: buildAuthHeaders(),
    });

    return parseCatalogResponse(response);
  },
  async create(payload: CatalogPayload): Promise<CatalogSummary> {
    const response = await fetch(`${API_URL}/catalogs`, {
      method: 'POST',
      headers: buildAuthHeaders(),
      body: JSON.stringify(payload),
    });

    return parseSingleCatalog(response);
  },
  async update(id: string, payload: CatalogPayload): Promise<CatalogSummary> {
    const response = await fetch(`${API_URL}/catalogs/${id}`, {
      method: 'PUT',
      headers: buildAuthHeaders(),
      body: JSON.stringify(payload),
    });

    return parseSingleCatalog(response);
  },
};
