import jwt from 'jsonwebtoken';
import { createError } from './errorHandler.js';
export const authenticate = async (req, res, next) => {
    try {
        const authHeader = req.headers.authorization;
        if (!authHeader || !authHeader.startsWith('Bearer ')) {
            throw createError('No token provided', 401);
        }
        const token = authHeader.substring(7);
        const secret = process.env.JWT_SECRET;
        if (!secret) {
            throw createError('JWT secret not configured', 500);
        }
        const decoded = jwt.verify(token, secret);
        req.user = decoded;
        next();
    }
    catch (error) {
        if (error instanceof jwt.JsonWebTokenError) {
            next(createError('Invalid token', 401));
        }
        else if (error instanceof jwt.TokenExpiredError) {
            next(createError('Token expired', 401));
        }
        else {
            next(error);
        }
    }
};
export const authorize = (...roles) => {
    return (req, res, next) => {
        if (!req.user) {
            return next(createError('Not authenticated', 401));
        }
        if (!roles.includes(req.user.role)) {
            return next(createError('Not authorized to access this resource', 403));
        }
        next();
    };
};
//# sourceMappingURL=auth.js.map